
import gurobi

def solve_optimization_problem():
    # Create a new Gurobi model
    model = gurobi.Model()

    # Define the variables
    knishes = model.addVar(vtype=gurobi.GRB.INTEGER, name="knishes")
    protein_bars = model.addVar(vtype=gurobi.GRB.CONTINUOUS, name="protein_bars")

    # Objective function
    model.setObjective(1.71 * knishes**2 + 1.41 * knishes * protein_bars + 6.44 * protein_bars**2, gurobi.GRB.MAXIMIZE)

    # Constraints
    # knishes contain 5 milligrams of iron
    # protein bars contain 5 milligrams of iron
    # These are implicit in the other constraints

    # At least 46 milligrams of iron must come from knishes and protein bars squared
    model.addConstraint(5 * knishes + 5 * protein_bars >= 46, name="iron_from_sources")

    # Correct interpretation of the constraint: 
    # At least 46 milligrams of iron must come from knishes squared and protein bars squared
    model.addConstraint(25 * knishes**2 + 25 * protein_bars**2 >= 46, name="iron_from_sources_squared")

    # 6 times the number of knishes, plus -3 times the number of protein bars should be at minimum zero
    model.addConstraint(6 * knishes - 3 * protein_bars >= 0, name="knishes_protein_bars_ratio")

    # At most 52 milligrams of iron from knishes squared plus protein bars squared
    model.addConstraint(25 * knishes**2 + 25 * protein_bars**2 <= 52, name="max_iron_from_sources_squared")

    # At most 52 milligrams of iron from knishes plus protein bars
    model.addConstraint(5 * knishes + 5 * protein_bars <= 52, name="max_iron_from_sources")

    # Solve the problem
    model.optimize()

    # Print the solution
    if model.status == gurobi.GRB.OPTIMAL:
        print("Optimal solution found.")
        print(f"Knishes: {knishes.varValue}")
        print(f"Protein Bars: {protein_bars.varValue}")
        print(f"Objective: {model.objVal}")
    elif model.status == gurobi.GRB.INFEASIBLE:
        print("The problem is infeasible.")
    else:
        print("The problem has no optimal solution.")

solve_optimization_problem()
