
import gurobipy as gp

# Create a new model
m = gp.Model("vitamin_optimization")

# Create variables
vitamin_d = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_d")
vitamin_b9 = m.addVar(vtype=gp.GRB.CONTINUOUS, name="vitamin_b9")
zinc = m.addVar(vtype=gp.GRB.CONTINUOUS, name="zinc")

# Set objective function
m.setObjective(3.53 * vitamin_d + 5.16 * vitamin_b9 + 2.52 * zinc, gp.GRB.MINIMIZE)

# Add constraints

# Resource constraints (using provided resource data)
resources = {
    'r0': {'description': 'immune support index', 'upper_bound': 198, 'x0': 5, 'x1': 6, 'x2': 11},
    'r1': {'description': 'digestive support index', 'upper_bound': 135, 'x0': 1, 'x1': 4, 'x2': 2},
    'r2': {'description': 'energy stability index', 'upper_bound': 174, 'x0': 2, 'x1': 7, 'x2': 11}
}

for r_id, r_data in resources.items():
    m.addConstr(r_data['x0'] * vitamin_d + r_data['x1'] * vitamin_b9 + r_data['x2'] * zinc <= r_data['upper_bound'], name=f"{r_data['description']}_upper_bound")


# Additional constraints from the problem description
m.addConstr(6 * vitamin_b9 + 11 * zinc >= 36, "combined_immune_b9_zinc")
m.addConstr(5 * vitamin_d + 6 * vitamin_b9 + 11 * zinc >= 36, "combined_immune_all")
m.addConstr(1 * vitamin_d + 4 * vitamin_b9 >= 44, "combined_digestive_d_b9")
m.addConstr(1 * vitamin_d + 4 * vitamin_b9 + 2 * zinc >= 44, "combined_digestive_all")
m.addConstr(2 * vitamin_d + 7 * vitamin_b9 >= 20, "combined_energy_d_b9")
m.addConstr(2 * vitamin_d + 11 * zinc >= 33, "combined_energy_d_zinc")
m.addConstr(2 * vitamin_d + 7 * vitamin_b9 + 11 * zinc >= 33, "combined_energy_all")
m.addConstr(7 * vitamin_d - 10 * zinc >= 0, "constraint_d_zinc")
m.addConstr(9 * vitamin_b9 - 3 * zinc >= 0, "constraint_b9_zinc")
m.addConstr(-5 * vitamin_d + 8 * vitamin_b9 >= 0, "constraint_d_b9")
m.addConstr(5 * vitamin_d + 6 * vitamin_b9 <= 150, "combined_immune_d_b9_upper")
m.addConstr(2 * vitamin_d + 11 * zinc <= 164, "combined_energy_d_zinc_upper")
m.addConstr(7 * vitamin_b9 + 11 * zinc <= 173, "combined_energy_b9_zinc_upper")


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('Vitamin D:', vitamin_d.x)
    print('Vitamin B9:', vitamin_b9.x)
    print('Zinc:', zinc.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('The model is infeasible.')
else:
    print(f"Optimization terminated with status {m.status}")

