
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("nutrition_optimization")

    # Create variables
    protein = m.addVar(vtype=gp.GRB.CONTINUOUS, name="protein")
    vitamin_b2 = m.addVar(vtype=gp.GRB.CONTINUOUS, name="vitamin_b2")
    carbs = m.addVar(vtype=gp.GRB.CONTINUOUS, name="carbs")

    # Set objective function
    m.setObjective(1 * protein + 6 * vitamin_b2 + 2 * carbs, gp.GRB.MAXIMIZE)

    # Add constraints
    m.addConstr(3.48 * protein + 0.04 * carbs >= 28, "muscle_growth_protein_carbs_min")
    m.addConstr(3.48 * protein + 3.43 * vitamin_b2 >= 34, "muscle_growth_protein_b2_min")
    m.addConstr(3.0 * protein + 3.44 * vitamin_b2 >= 25, "kidney_support_protein_b2_min")
    m.addConstr(0.28 * protein + 2.49 * vitamin_b2 >= 54, "energy_stability_protein_b2_min")
    m.addConstr(6 * vitamin_b2 - 5 * carbs >= 0, "vitamin_b2_carbs_relation")
    m.addConstr(3.43 * vitamin_b2 + 0.04 * carbs <= 129, "muscle_growth_b2_carbs_max")
    m.addConstr(3.48 * protein + 0.04 * carbs <= 176, "muscle_growth_protein_carbs_max")
    m.addConstr(3.48 * protein + 3.43 * vitamin_b2 + 0.04 * carbs <= 176, "muscle_growth_all_max")
    m.addConstr(3.44 * vitamin_b2 + 3.52 * carbs <= 128, "kidney_support_b2_carbs_max")
    m.addConstr(3.0 * protein + 3.52 * carbs <= 61, "kidney_support_protein_carbs_max")
    m.addConstr(3.0 * protein + 3.44 * vitamin_b2 + 3.52 * carbs <= 61, "kidney_support_all_max")
    m.addConstr(2.49 * vitamin_b2 + 1.22 * carbs <= 157, "energy_stability_b2_carbs_max")
    m.addConstr(0.28 * protein + 2.49 * vitamin_b2 <= 79, "energy_stability_protein_b2_max")
    m.addConstr(0.28 * protein + 2.49 * vitamin_b2 + 1.22 * carbs <= 79, "energy_stability_all_max")


    # Resource constraints (from the provided resource dictionary)
    m.addConstr(3.48 * protein + 3.43 * vitamin_b2 + 0.04 * carbs <= 202, "r0_muscle_growth_max")
    m.addConstr(3.0 * protein + 3.44 * vitamin_b2 + 3.52 * carbs <= 144, "r1_kidney_support_max")
    m.addConstr(0.28 * protein + 2.49 * vitamin_b2 + 1.22 * carbs <= 201, "r2_energy_stability_max")


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('Protein: %g' % protein.x)
        print('Vitamin B2: %g' % vitamin_b2.x)
        print('Carbs: %g' % carbs.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
