
import gurobipy as gp

# Create a new model
m = gp.Model("vitamin_optimization")

# Create variables
vitamin_e = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_e")
magnesium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="magnesium")
vitamin_a = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_a")

# Set objective function
m.setObjective(2 * vitamin_e + 1 * magnesium + 6 * vitamin_a, gp.GRB.MAXIMIZE)

# Add constraints based on resource indices
resource_constraints = {
    'r0': {'var_coeffs': [13, 10, 1], 'ub': 178},  # kidney support
    'r1': {'var_coeffs': [10, 22, 4], 'ub': 162},  # energy stability
    'r2': {'var_coeffs': [5, 17, 2], 'ub': 103},  # digestive support
    'r3': {'var_coeffs': [7, 20, 5], 'ub': 208},  # cognitive performance
    'r4': {'var_coeffs': [8, 4, 17], 'ub': 127},  # muscle growth
}

for r, data in resource_constraints.items():
    m.addConstr(
        data['var_coeffs'][0] * vitamin_e + data['var_coeffs'][1] * magnesium + data['var_coeffs'][2] * vitamin_a <= data['ub'], name=r
    )


# Add additional constraints
m.addConstr(13 * vitamin_e + 10 * magnesium >= 29, "kidney_support_min_e_mg")
m.addConstr(10 * vitamin_e + 22 * magnesium >= 52, "energy_stability_min_e_mg")
m.addConstr(10 * vitamin_e + 4 * vitamin_a >= 40, "energy_stability_min_e_a")
m.addConstr(17 * magnesium + 2 * vitamin_a >= 31, "digestive_support_min_mg_a")
m.addConstr(4 * magnesium + 17 * vitamin_a >= 42, "muscle_growth_min_mg_a")
m.addConstr(8 * vitamin_e + 17 * vitamin_a >= 16, "muscle_growth_min_e_a")
m.addConstr(13 * vitamin_e + 10 * magnesium <= 99, "kidney_support_max_e_mg")
m.addConstr(13 * vitamin_e + 1 * vitamin_a <= 104, "kidney_support_max_e_a")
m.addConstr(13 * vitamin_e + 10 * magnesium + 1 * vitamin_a <= 161, "kidney_support_max_all_1")
m.addConstr(13 * vitamin_e + 10 * magnesium + 1 * vitamin_a <= 161, "kidney_support_max_all_2")  # Duplicate constraint
m.addConstr(10 * vitamin_e + 4 * vitamin_a <= 130, "energy_stability_max_e_a")
m.addConstr(22 * magnesium + 4 * vitamin_a <= 130, "energy_stability_max_mg_a")
m.addConstr(10 * vitamin_e + 22 * magnesium + 4 * vitamin_a <= 130, "energy_stability_max_all")
m.addConstr(5 * vitamin_e + 2 * vitamin_a <= 72, "digestive_support_max_e_a")
m.addConstr(17 * magnesium + 2 * vitamin_a <= 73, "digestive_support_max_mg_a")
m.addConstr(5 * vitamin_e + 17 * magnesium + 2 * vitamin_a <= 73, "digestive_support_max_all")
m.addConstr(20 * magnesium + 5 * vitamin_a <= 183, "cognitive_performance_max_mg_a")
m.addConstr(7 * vitamin_e + 20 * magnesium + 5 * vitamin_a <= 151, "cognitive_performance_max_all_1")
m.addConstr(7 * vitamin_e + 20 * magnesium + 5 * vitamin_a <= 151, "cognitive_performance_max_all_2") # Duplicate constraint
m.addConstr(8 * vitamin_e + 17 * vitamin_a <= 127, "muscle_growth_max_e_a")
m.addConstr(8 * vitamin_e + 4 * magnesium <= 51, "muscle_growth_max_e_mg")
m.addConstr(4 * magnesium + 17 * vitamin_a <= 106, "muscle_growth_max_mg_a")
m.addConstr(8 * vitamin_e + 4 * magnesium + 17 * vitamin_a <= 106, "muscle_growth_max_all")


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('Vitamin E: %g' % vitamin_e.x)
    print('Magnesium: %g' % magnesium.x)
    print('Vitamin A: %g' % vitamin_a.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('The model is infeasible.')
else:
    print('Optimization ended with status %d' % m.status)
