
import gurobipy as gp

# Create a new model
m = gp.Model("vitamin_optimization")

# Create variables
b5 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="B5")  # milligrams of vitamin B5
b3 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="B3")  # milligrams of vitamin B3
b2 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="B2")  # milligrams of vitamin B2

# Set objective function
m.setObjective(6 * b5 + 3 * b3 + 5 * b2, gp.GRB.MINIMIZE)

# Add constraints
m.addConstr(0.06 * b5 + 0.26 * b2 >= 11, "immune_b5_b2_min")
m.addConstr(0.06 * b5 + 8.56 * b3 + 0.26 * b2 >= 15, "immune_total_min")
m.addConstr(1.05 * b5 + 7.92 * b3 >= 11, "cognitive_b5_b3_min")
m.addConstr(1.05 * b5 + 7.92 * b3 + 4.16 * b2 >= 12, "cognitive_total_min")
m.addConstr(8.57 * b5 + 7.38 * b3 >= 25, "muscle_b5_b3_min")
m.addConstr(8.57 * b5 + 7.38 * b3 + 6.67 * b2 >= 25, "muscle_total_min")
m.addConstr(2 * b5 - 9 * b2 >= 0, "b5_b2_relation")
m.addConstr(8.56 * b3 + 0.26 * b2 <= 43, "immune_b3_b2_max")
m.addConstr(0.06 * b5 + 0.26 * b2 <= 19, "immune_b5_b2_max")
m.addConstr(1.05 * b5 + 7.92 * b3 + 4.16 * b2 <= 58, "cognitive_total_max")


# Resource Constraints (from the provided resource dictionary)
resources = {'r0': {'description': 'immune support index', 'upper_bound': 50, 'x0': 0.06, 'x1': 8.56, 'x2': 0.26}, 
            'r1': {'description': 'cognitive performance index', 'upper_bound': 66, 'x0': 1.05, 'x1': 7.92, 'x2': 4.16}, 
            'r2': {'description': 'muscle growth index', 'upper_bound': 126, 'x0': 8.57, 'x1': 7.38, 'x2': 6.67}}

for r_key, r_data in resources.items():
    m.addConstr(r_data['x0'] * b5 + r_data['x1'] * b3 + r_data['x2'] * b2 <= r_data['upper_bound'], r_key)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Optimal objective: %g' % m.objVal)
    print('B5:', b5.x)
    print('B3:', b3.x)
    print('B2:', b2.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('Model is infeasible')
else:
    print('Optimization ended with status %d' % m.status)

