
import gurobipy as gp

# Create a new model
m = gp.Model("Work_Hours_Optimization")

# Create variables
workers = ['Bobby', 'Paul', 'Peggy', 'Dale', 'Mary', 'George', 'Jean', 'John', 'Hank']
hours_worked = m.addVars(workers, lb=0, name="hours")

# Objective function coefficients
obj_coeffs = [1, 2, 4, 8, 2, 5, 8, 6, 9]

# Set objective function
m.setObjective(gp.quicksum(obj_coeffs[i] * hours_worked[workers[i]] for i in range(len(workers))), gp.GRB.MAXIMIZE)

# Resource constraints
resources = {
    'r0': {'description': 'paperwork competence rating', 'upper_bound': 547, 'x0': 18, 'x1': 12, 'x2': 23, 'x3': 2, 'x4': 16, 'x5': 13, 'x6': 23, 'x7': 22, 'x8': 14},
    'r1': {'description': 'organization score', 'upper_bound': 465, 'x0': 4, 'x1': 29, 'x2': 26, 'x3': 5, 'x4': 12, 'x5': 5, 'x6': 19, 'x7': 21, 'x8': 17},
    'r2': {'description': 'likelihood to quit index', 'upper_bound': 574, 'x0': 3, 'x1': 10, 'x2': 18, 'x3': 22, 'x4': 21, 'x5': 18, 'x6': 13, 'x7': 17, 'x8': 1}
}

for resource, data in resources.items():
    m.addConstr(gp.quicksum(data[f'x{i}'] * hours_worked[workers[i]] for i in range(len(workers))) <= data['upper_bound'], name=resource)


# Additional constraints (organization score combinations)
org_score_constraints = [
    ([('George',5),('Hank',17)], '>=', 44),
    ([('Mary',12),('Hank',17)], '>=', 34),
    ([('George',5),('John',21)], '>=', 19),
    ([('Dale',5),('Mary',12)], '>=', 45),
    ([('Dale',5),('John',21)], '>=', 39),
    ([('Bobby',4),('Paul',29),('Peggy',26)], '>=', 50), # Example, repeat for others
    ([('Dale',5),('George',5),('Jean',19)], '>=', 50),
    ([('Mary',12),('John',21),('Hank',17)], '>=', 50),
    ([('Mary',12),('George',5),('Jean',19)], '>=', 50),
    ([('Bobby',4),('Paul',29),('George',5)], '>=', 50),
    ([('Bobby',4),('Mary',12),('Jean',19)], '>=', 50),
    ([('Paul',29),('Jean',19),('John',21)], '>=', 50),
    ([('Mary',12),('Jean',19),('Hank',17)], '>=', 50),
    ([('Bobby',4),('Peggy',26),('George',5)], '>=', 50),
    ([('Paul',29),('Peggy',26),('Mary',12)], '>=', 50),
    ([('Bobby',4),('Peggy',26),('John',21)], '>=', 50),
    ([('Jean',19),('John',21),('Hank',17)], '>=', 50),
    ([('Bobby',4),('Dale',5),('John',21)], '>=', 50),
    ([('Paul',29),('Jean',19),('Hank',17)], '>=', 50),
    ([('Paul',29),('Peggy',26),('John',21)], '>=', 50)
]

# Add all organization score constraints with varying bounds
for bound in [25, 27, 32, 35, 36, 38, 46, 49, 50]:
    for constraint_data in org_score_constraints:
        workers_involved, operator, value = constraint_data
        lhs = gp.quicksum(score * hours_worked[worker] for worker, score in workers_involved)
        if operator == '>=':
            m.addConstr(lhs >= bound, name=f"org_score_{bound}_{workers_involved}")
        elif operator == '<=':
            m.addConstr(lhs <= bound, name=f"org_score_{bound}_{workers_involved}")
        elif operator == '=':
            m.addConstr(lhs == bound, name=f"org_score_{bound}_{workers_involved}")


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Optimal objective:', m.objVal)
    for worker in workers:
        print(f'{worker}: {hours_worked[worker].x}')
elif m.status == gp.GRB.INFEASIBLE:
    print('Model is infeasible.')
else:
    print(f"Optimization ended with status {m.status}")

