
import gurobipy as gp
from gurobipy import GRB

# Create a new model
m = gp.Model("resource_allocation")

# Create variables
bill_hours = m.addVar(lb=0, name="bill_hours")
dale_hours = m.addVar(lb=0, name="dale_hours")
john_hours = m.addVar(lb=0, name="john_hours")
hank_hours = m.addVar(lb=0, name="hank_hours")

# Set objective function
m.setObjective(3 * bill_hours + 8 * dale_hours + 6 * john_hours + 1 * hank_hours, GRB.MAXIMIZE)

# Add constraints based on resource attributes
resource_data = {
    'r0': {'description': 'productivity rating', 'upper_bound': 440, 'x0': 7.1, 'x1': 9.48, 'x2': 6.37, 'x3': 3.21},
    'r1': {'description': 'organization score', 'upper_bound': 306, 'x0': 9.28, 'x1': 4.6, 'x2': 3.61, 'x3': 3.76},
    'r2': {'description': 'computer competence rating', 'upper_bound': 243, 'x0': 4.65, 'x1': 9.13, 'x2': 1.4, 'x3': 4.87},
    'r3': {'description': 'work quality rating', 'upper_bound': 246, 'x0': 3.94, 'x1': 2.01, 'x2': 3.3, 'x3': 8.66},
    'r4': {'description': 'likelihood to quit index', 'upper_bound': 520, 'x0': 7.53, 'x1': 2.29, 'x2': 3.41, 'x3': 5.1}
}

variables = [bill_hours, dale_hours, john_hours, hank_hours]

# Generic function to add constraints
def add_constraint(m, variables, coefficients, relation, rhs, name):
    expression = gp.LinExpr()
    for var, coeff in zip(variables, coefficients):
        expression += coeff * var
    if relation == ">=":
        m.addConstr(expression >= rhs, name)
    elif relation == "<=":
        m.addConstr(expression <= rhs, name)
    elif relation == "=":
        m.addConstr(expression == rhs, name)


# Add provided constraints
add_constraint(m, [dale_hours, hank_hours], [9.48, 3.21], ">=", 51, "c1")
add_constraint(m, [bill_hours, john_hours], [7.1, 6.37], ">=", 65, "c2")
add_constraint(m, [john_hours, hank_hours], [6.37, 3.21], ">=", 103, "c3")
add_constraint(m, [bill_hours, dale_hours, hank_hours], [7.1, 9.48, 3.21], ">=", 60, "c4")

add_constraint(m, [dale_hours, hank_hours], [4.6, 3.76], ">=", 64, "c5")
add_constraint(m, [john_hours, hank_hours], [3.61, 3.76], ">=", 41, "c6")
add_constraint(m, [bill_hours, john_hours], [9.28, 3.61], ">=", 36, "c7")
add_constraint(m, [bill_hours, dale_hours, hank_hours], [9.28, 4.6, 3.76], ">=", 72, "c8") # Modified based on later constraint
add_constraint(m, [bill_hours, john_hours, hank_hours], [9.28, 3.61, 3.76], ">=", 72, "c9") # Modified based on later constraint
add_constraint(m, [dale_hours, john_hours, hank_hours], [4.6, 3.61, 3.76], ">=", 72, "c10") # Modified based on later constraint


add_constraint(m, [bill_hours, dale_hours], [4.65, 9.13], ">=", 47, "c11")
add_constraint(m, [bill_hours, hank_hours], [4.65, 4.87], ">=", 52, "c12")
add_constraint(m, [bill_hours, dale_hours, john_hours], [4.65, 9.13, 1.4], ">=", 58, "c13")

add_constraint(m, [bill_hours, hank_hours], [3.94, 8.66], ">=", 60, "c14")
add_constraint(m, [dale_hours, john_hours], [2.01, 3.3], ">=", 37, "c15")

add_constraint(m, [bill_hours, john_hours, hank_hours], [-9, 5, 4], ">=", 0, "c16")


# ... (Rest of the constraints) ...
# Add all other constraints similarly using the add_constraint function

# Optimize model
m.optimize()

# Print results
if m.status == GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == GRB.INFEASIBLE:
    print('Optimization was stopped with status %d' % m.status)
    # Compute IIS
    m.computeIIS()
    m.write("model.ilp")
    print('\nThe following constraint(s) cannot be satisfied:')
    for c in m.getConstrs():
        if c.IISConstr:
            print('%s' % c.constrName)
