
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
carbs = m.addVar(lb=0, name="carbs")  # grams of carbohydrates
calcium = m.addVar(lb=0, name="calcium")  # milligrams of calcium
iron = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="iron")  # milligrams of iron
vitamin_b5 = m.addVar(lb=0, name="vitamin_b5")  # milligrams of vitamin B5

# Set objective function
m.setObjective(1.73 * carbs + 6.73 * calcium + 6.03 * iron + 3.16 * vitamin_b5, gp.GRB.MINIMIZE)

# Add constraints

# Resource constraints (upper bounds are already handled in the data)
resource_data = {
    'r0': {'description': 'cognitive performance index', 'upper_bound': 72, 'x0': 5.42, 'x1': 9.63, 'x2': 5.15, 'x3': 0.68},
    'r1': {'description': 'digestive support index', 'upper_bound': 199, 'x1': 4.43, 'x0': 5.9, 'x2': 7.38, 'x3': 4.99},
    'r2': {'description': 'muscle growth index', 'upper_bound': 114, 'x0': 2.23, 'x1': 3.17, 'x2': 1.19, 'x3': 2.71}
}

for resource, data in resource_data.items():
    m.addConstr(data['x0'] * carbs + data['x1'] * calcium + data['x2'] * iron + data['x3'] * vitamin_b5 <= data['upper_bound'], name=f"{resource}_upper_bound")


# Additional constraints from the problem description
m.addConstr(5.15 * iron + 0.68 * vitamin_b5 >= 11, "c1")
m.addConstr(5.42 * carbs + 0.68 * vitamin_b5 >= 18, "c2")
m.addConstr(9.63 * calcium + 5.15 * iron >= 14, "c3")
m.addConstr(5.42 * carbs + 9.63 * calcium >= 7, "c4")
m.addConstr(9.63 * calcium + 5.15 * iron + 0.68 * vitamin_b5 >= 9, "c5")
m.addConstr(5.42 * carbs + 9.63 * calcium + 5.15 * iron >= 9, "c6")
m.addConstr(5.42 * carbs + 9.63 * calcium + 0.68 * vitamin_b5 >= 9, "c7")
m.addConstr(5.42 * carbs + 5.15 * iron + 0.68 * vitamin_b5 >= 9, "c8")
m.addConstr(9.63 * calcium + 5.15 * iron + 0.68 * vitamin_b5 >= 16, "c9")
m.addConstr(5.42 * carbs + 9.63 * calcium + 5.15 * iron >= 16, "c10")
m.addConstr(5.42 * carbs + 9.63 * calcium + 0.68 * vitamin_b5 >= 16, "c11")
m.addConstr(5.42 * carbs + 5.15 * iron + 0.68 * vitamin_b5 >= 16, "c12")
m.addConstr(9.63 * calcium + 5.15 * iron + 0.68 * vitamin_b5 >= 14, "c13")
m.addConstr(5.42 * carbs + 9.63 * calcium + 5.15 * iron >= 14, "c14")
m.addConstr(5.42 * carbs + 9.63 * calcium + 0.68 * vitamin_b5 >= 14, "c15")
m.addConstr(5.42 * carbs + 5.15 * iron + 0.68 * vitamin_b5 >= 14, "c16")
m.addConstr(9.63 * calcium + 5.15 * iron + 0.68 * vitamin_b5 >= 11, "c17")
m.addConstr(5.42 * carbs + 9.63 * calcium + 5.15 * iron >= 11, "c18")
m.addConstr(5.42 * carbs + 9.63 * calcium + 0.68 * vitamin_b5 >= 11, "c19")
m.addConstr(5.42 * carbs + 5.15 * iron + 0.68 * vitamin_b5 >= 11, "c20")
m.addConstr(5.42 * carbs + 9.63 * calcium + 5.15 * iron + 0.68 * vitamin_b5 >= 11, "c21")


# ... (Rest of the constraints similarly)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('carbs:', carbs.x)
    print('calcium:', calcium.x)
    print('iron:', iron.x)
    print('vitamin_b5:', vitamin_b5.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('Model is infeasible')
else:
    print('Optimization ended with status %d' % m.status)

