
from gurobipy import Model, GRB

# Create a new model
m = Model("nutrition_optimization")

# Create variables
vitamin_k = m.addVar(lb=0, name="vitamin_k")  # milligrams of vitamin K
fiber = m.addVar(lb=0, name="fiber")  # grams of fiber

# Set objective function
m.setObjective(7 * vitamin_k * fiber + 8 * fiber, GRB.MINIMIZE)

# Add constraints

# Resource constraints (using the provided dictionary would make this more scalable)
m.addConstr(10 * vitamin_k + 14 * fiber >= 12, "c_cognitive_lower")
m.addConstr(5 * vitamin_k + 15 * fiber >= 16, "c_muscle_lower")
m.addConstr(7 * vitamin_k**2 + 2 * fiber**2 >= 30, "c_digestive_lower")
m.addConstr(17 * vitamin_k**2 + 8 * fiber**2 >= 18, "c_immune_lower")
m.addConstr(-3 * vitamin_k + 5 * fiber >= 0, "c_vitamin_fiber_ratio")


# Upper bound constraints (added based on the problem description)
m.addConstr(10 * vitamin_k**2 + 14 * fiber**2 <= 31, "c_cognitive_upper") # squared terms as specified
m.addConstr(5 * vitamin_k + 15 * fiber <= 19, "c_muscle_upper")
m.addConstr(7 * vitamin_k**2 + 2 * fiber**2 <= 76, "c_digestive_upper") # squared terms as specified
m.addConstr(17 * vitamin_k + 8 * fiber <= 41, "c_immune_upper")


# Optimize model
m.optimize()

# Print results
if m.status == GRB.OPTIMAL:
    print(f"Optimal solution found:")
    print(f"  Milligrams of Vitamin K: {vitamin_k.x}")
    print(f"  Grams of Fiber: {fiber.x}")
    print(f"  Objective Value: {m.objVal}")
elif m.status == GRB.INFEASIBLE:
    print("Model is infeasible.")
else:
    print(f"Optimization terminated with status: {m.status}")

