
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    b2 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="B2")
    c = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="C")
    b7 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="B7")

    # Set objective function
    m.setObjective(9.96 * b2 + 9.49 * c + 2.05 * b7, gp.GRB.MINIMIZE)

    # Add constraints based on provided resource attributes
    kidney_support = {'B2': 4, 'C': 8, 'B7': 14}
    immune_support = {'B2': 9, 'C': 10, 'B7': 3}
    digestive_support = {'B2': 1, 'C': 8, 'B7': 4}
    energy_stability = {'B2': 7, 'C': 2, 'B7': 14}

    # Kidney support constraints
    m.addConstr(kidney_support['B2'] * b2 + kidney_support['C'] * c >= 16)
    m.addConstr(kidney_support['B2'] * b2 + kidney_support['B7'] * b7 >= 28)
    m.addConstr(kidney_support['B2'] * b2 + kidney_support['C'] * c + kidney_support['B7'] * b7 >= 28) # Consolidated

    # Immune support constraints
    m.addConstr(immune_support['C'] * c + immune_support['B7'] * b7 >= 20)
    m.addConstr(immune_support['B2'] * b2 + immune_support['B7'] * b7 >= 31)
    m.addConstr(immune_support['B2'] * b2 + immune_support['C'] * c + immune_support['B7'] * b7 >= 31) # Consolidated

    # Digestive support constraints (Resolved potential conflicts by taking the most restrictive constraints)
    m.addConstr(digestive_support['B2'] * b2 + digestive_support['B7'] * b7 >= 41)
    m.addConstr(digestive_support['B2'] * b2 + digestive_support['C'] * c >= 45)
    m.addConstr(digestive_support['B2'] * b2 + digestive_support['C'] * c <= 45) # Added upper bound
    m.addConstr(digestive_support['B2'] * b2 + digestive_support['B7'] * b7 <= 117) # Added upper bound
    m.addConstr(digestive_support['B2'] * b2 + digestive_support['C'] * c + digestive_support['B7'] * b7 >= 24)


    # Energy stability constraints (Resolved potential conflicts by taking the most restrictive constraints)
    m.addConstr(energy_stability['B2'] * b2 + energy_stability['C'] * c >= 16)
    m.addConstr(energy_stability['B2'] * b2 + energy_stability['B7'] * b7 >= 16)
    m.addConstr(energy_stability['B2'] * b2 + energy_stability['B7'] * b7 <= 25) # Added upper bound
    m.addConstr(energy_stability['B2'] * b2 + energy_stability['C'] * c + energy_stability['B7'] * b7 >= 16)

    # Additional constraints
    m.addConstr(-6 * b2 + c >= 0)
    m.addConstr(7 * c - 10 * b7 >= 0)


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('B2:', b2.x)
        print('C:', c.x)
        print('B7:', b7.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
