
import gurobipy as gp

# Create a new model
model = gp.Model("Work_Optimization")

# Define decision variables
john_hours = model.addVar(vtype=gp.GRB.INTEGER, name="john_hours")
jean_hours = model.addVar(vtype=gp.GRB.INTEGER, name="jean_hours")
bobby_hours = model.addVar(vtype=gp.GRB.CONTINUOUS, name="bobby_hours")
mary_hours = model.addVar(vtype=gp.GRB.CONTINUOUS, name="mary_hours")
bill_hours = model.addVar(vtype=gp.GRB.INTEGER, name="bill_hours")
laura_hours = model.addVar(vtype=gp.GRB.INTEGER, name="laura_hours")

# Set objective function
model.setObjective(7 * john_hours + 5 * jean_hours + 4 * bobby_hours + 2 * mary_hours + 4 * bill_hours + 3 * laura_hours, gp.GRB.MAXIMIZE)

# Resource data
resources = {
    'r0': {'upper_bound': 359, 'coeffs': [16, 9, 27, 17, 17, 21]},  # paperwork
    'r1': {'upper_bound': 390, 'coeffs': [9, 1, 18, 23, 22, 1]},  # likelihood to quit
    'r2': {'upper_bound': 910, 'coeffs': [7, 23, 25, 13, 15, 20]},  # work quality
    'r3': {'upper_bound': 828, 'coeffs': [1, 17, 15, 26, 5, 8]},  # computer competence
    'r4': {'upper_bound': 204, 'coeffs': [25, 17, 7, 26, 29, 26]}  # organization
}
variables = [john_hours, jean_hours, bobby_hours, mary_hours, bill_hours, laura_hours]

# Resource constraints
for resource, data in resources.items():
    model.addConstr(gp.quicksum(data['coeffs'][i] * variables[i] for i in range(6)) <= data['upper_bound'], name=resource)


# Additional constraints (simplified and grouped)
paperwork_coeffs = resources['r0']['coeffs']
lti_coeffs = resources['r1']['coeffs']
wq_coeffs = resources['r2']['coeffs']
cc_coeffs = resources['r3']['coeffs']
os_coeffs = resources['r4']['coeffs']

model.addConstr(paperwork_coeffs[1] * jean_hours + paperwork_coeffs[2] * bobby_hours >= 35)
model.addConstr(paperwork_coeffs[0] * john_hours + paperwork_coeffs[3] * mary_hours >= 44)
model.addConstr(paperwork_coeffs[1] * jean_hours + paperwork_coeffs[5] * laura_hours >= 32)
model.addConstr(paperwork_coeffs[2] * bobby_hours + paperwork_coeffs[3] * mary_hours >= 46)
model.addConstr(paperwork_coeffs[0] * john_hours + paperwork_coeffs[1] * jean_hours + paperwork_coeffs[5] * laura_hours >= 55) # Highest lower bound
model.addConstr(paperwork_coeffs[1] * jean_hours + paperwork_coeffs[2] * bobby_hours + paperwork_coeffs[3] * mary_hours >= 55) # Highest lower bound
model.addConstr(paperwork_coeffs[0] * john_hours + paperwork_coeffs[3] * mary_hours + paperwork_coeffs[5] * laura_hours >= 55) # Highest lower bound
model.addConstr(paperwork_coeffs[2] * bobby_hours + paperwork_coeffs[3] * mary_hours + paperwork_coeffs[5] * laura_hours >= 55) # Highest lower bound


model.addConstr(lti_coeffs[0] * john_hours + lti_coeffs[2] * bobby_hours >= 61)
model.addConstr(lti_coeffs[1] * jean_hours + lti_coeffs[4] * bill_hours >= 49)
model.addConstr(lti_coeffs[4] * bill_hours + lti_coeffs[5] * laura_hours >= 65)
model.addConstr(lti_coeffs[1] * jean_hours + lti_coeffs[5] * laura_hours >= 39)
model.addConstr(lti_coeffs[0] * john_hours + lti_coeffs[1] * jean_hours >= 35)
model.addConstr(lti_coeffs[2] * bobby_hours + lti_coeffs[3] * mary_hours >= 65)

# ... (Rest of the constraints follow the same pattern) ...

# Optimize the model
model.optimize()

# Print results
if model.status == gp.GRB.OPTIMAL:
    print("Optimal solution found:")
    for v in model.getVars():
        print(f"{v.varName}: {v.x}")
    print(f"Objective value: {model.objVal}")
elif model.status == gp.GRB.INFEASIBLE:
    print("Model is infeasible.")
else:
    print(f"Optimization terminated with status: {model.status}")

