
import gurobipy as gp

# Create a new model
m = gp.Model("optimization_problem")

# Create variables
vitamin_e = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_e")
protein = m.addVar(vtype=gp.GRB.CONTINUOUS, name="protein")
zinc = m.addVar(vtype=gp.GRB.CONTINUOUS, name="zinc")

# Set objective function
m.setObjective(7.61 * vitamin_e + 3.07 * protein + 6.17 * zinc, gp.GRB.MINIMIZE)

# Add constraints based on resource attributes
resources = {
    'r0': {'description': 'digestive support index', 'upper_bound': 91, 'x0': 3, 'x1': 14, 'x2': 14},
    'r1': {'description': 'cardiovascular support index', 'upper_bound': 66, 'x0': 5, 'x1': 12, 'x2': 2},
    'r2': {'description': 'muscle growth index', 'upper_bound': 102, 'x0': 3, 'x1': 7, 'x2': 3}
}

for r_id, r_data in resources.items():
    m.addConstr(r_data['x0'] * vitamin_e + r_data['x1'] * protein + r_data['x2'] * zinc <= r_data['upper_bound'], name=f"{r_data['description']}_upper_bound")


# Add additional constraints
m.addConstr(3 * vitamin_e + 14 * zinc >= 24, "digestive_support_1")
m.addConstr(14 * protein + 14 * zinc >= 30, "digestive_support_2")
m.addConstr(3 * vitamin_e + 14 * protein >= 17, "digestive_support_3")
m.addConstr(3 * vitamin_e + 14 * protein + 14 * zinc >= 17, "digestive_support_4")

m.addConstr(5 * vitamin_e + 2 * zinc >= 8, "cardiovascular_support_1")
m.addConstr(5 * vitamin_e + 12 * protein + 2 * zinc >= 8, "cardiovascular_support_2")

m.addConstr(7 * protein + 3 * zinc >= 21, "muscle_growth_1")
m.addConstr(3 * vitamin_e + 7 * protein >= 13, "muscle_growth_2")
m.addConstr(3 * vitamin_e + 7 * protein + 3 * zinc >= 20, "muscle_growth_3")
m.addConstr(3 * vitamin_e + 7 * protein + 3 * zinc >= 20, "muscle_growth_4")  # Duplicate constraint

m.addConstr(5 * protein - 8 * zinc >= 0, "protein_zinc_relation")
m.addConstr(8 * vitamin_e - 1 * zinc >= 0, "vitamin_e_zinc_relation")

m.addConstr(5 * vitamin_e + 2 * zinc <= 59, "cardiovascular_support_upper_bound")
m.addConstr(3 * vitamin_e + 3 * zinc <= 96, "muscle_growth_upper_bound")


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('Vitamin E:', vitamin_e.x)
    print('Protein:', protein.x)
    print('Zinc:', zinc.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('Optimization problem is infeasible.')
else:
    print(f"Optimization terminated with status {m.status}")

