
import gurobipy as gp

# Create a new model
m = gp.Model("vitamin_optimization")

# Create variables
vitamin_d = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_d")
potassium = m.addVar(vtype=gp.GRB.CONTINUOUS, name="potassium")
vitamin_k = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_k")

# Set objective function
m.setObjective(8 * vitamin_d + 3 * potassium + 1 * vitamin_k, gp.GRB.MINIMIZE)

# Add constraints based on provided resource attributes
resources = {
    'r0': {'description': 'digestive support index', 'upper_bound': 22, 'x0': 3, 'x1': 3, 'x2': 1},
    'r1': {'description': 'cardiovascular support index', 'upper_bound': 50, 'x0': 3, 'x1': 5, 'x2': 5},
    'r2': {'description': 'cognitive performance index', 'upper_bound': 18, 'x0': 3, 'x1': 2, 'x2': 5},
    'r3': {'description': 'immune support index', 'upper_bound': 19, 'x0': 1, 'x1': 1, 'x2': 4},
    'r4': {'description': 'energy stability index', 'upper_bound': 25, 'x0': 1, 'x1': 5, 'x2': 2}
}

for r_id, r_data in resources.items():
    m.addConstr(r_data['x0'] * vitamin_d + r_data['x1'] * potassium + r_data['x2'] * vitamin_k <= r_data['upper_bound'], name=f"{r_data['description']}_upper_bound")


# Add additional constraints from the problem description
m.addConstr(3 * vitamin_d + 3 * potassium >= 2, "digestive_support_1")
m.addConstr(3 * potassium + 1 * vitamin_k >= 7, "digestive_support_2")
m.addConstr(3 * vitamin_d + 3 * potassium + 1 * vitamin_k >= 7, "digestive_support_3")

m.addConstr(5 * potassium + 5 * vitamin_k >= 6, "cardiovascular_support_1")
m.addConstr(3 * vitamin_d + 5 * vitamin_k >= 9, "cardiovascular_support_2")
m.addConstr(3 * vitamin_d + 5 * potassium + 5 * vitamin_k >= 9, "cardiovascular_support_3")

m.addConstr(3 * vitamin_d + 5 * vitamin_k >= 3, "cognitive_performance_1")
m.addConstr(2 * potassium + 5 * vitamin_k >= 4, "cognitive_performance_2")
m.addConstr(3 * vitamin_d + 2 * potassium + 5 * vitamin_k >= 6, "cognitive_performance_3")
m.addConstr(3 * vitamin_d + 2 * potassium + 5 * vitamin_k <= 14, "cognitive_performance_4") # Added upper bound


m.addConstr(1 * potassium + 4 * vitamin_k >= 2, "immune_support_1")
m.addConstr(1 * vitamin_d + 4 * vitamin_k >= 6, "immune_support_2")
m.addConstr(1 * vitamin_d + 1 * potassium + 4 * vitamin_k >= 5, "immune_support_3")
m.addConstr(1 * vitamin_d + 1 * potassium + 4 * vitamin_k <= 10, "immune_support_4") # Added upper bound

m.addConstr(5 * potassium + 2 * vitamin_k >= 4, "energy_stability_1")
m.addConstr(1 * vitamin_d + 5 * potassium >= 2, "energy_stability_2")
m.addConstr(1 * vitamin_d + 5 * potassium + 2 * vitamin_k >= 8, "energy_stability_3")
m.addConstr(1 * vitamin_d + 2 * vitamin_k <= 15, "energy_stability_4") # Added upper bound
m.addConstr(5 * potassium + 2 * vitamin_k <= 13, "energy_stability_5") # Added upper bound


m.addConstr(-7 * vitamin_d + 8 * potassium >= 0, "custom_constraint_1")
m.addConstr(-7 * vitamin_d + 2 * vitamin_k >= 0, "custom_constraint_2")



# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('vitamin_d:', vitamin_d.x)
    print('potassium:', potassium.x)
    print('vitamin_k:', vitamin_k.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('The model is infeasible.')
else:
    print(f"Optimization ended with status {m.status}")
