
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    vitamin_c = m.addVar(vtype=gp.GRB.CONTINUOUS, name="vitamin_c")
    vitamin_b7 = m.addVar(vtype=gp.GRB.CONTINUOUS, name="vitamin_b7")

    # Set objective function
    m.setObjective(3.56 * vitamin_c + 9.33 * vitamin_b7, gp.GRB.MAXIMIZE)

    # Add constraints
    m.addConstr(5.67 * vitamin_c + 6.06 * vitamin_b7 >= 84, "cardiovascular_min")
    m.addConstr(0.73 * vitamin_c + 4.16 * vitamin_b7 >= 21, "digestive_min")
    m.addConstr(0.37 * vitamin_c + 1.26 * vitamin_b7 >= 48, "kidney_min")
    m.addConstr(3.56 * vitamin_c + 1.87 * vitamin_b7 >= 58, "energy_min")
    m.addConstr(5 * vitamin_c - 3 * vitamin_b7 >= 0, "vitamin_ratio")

    m.addConstr(5.67 * vitamin_c + 6.06 * vitamin_b7 <= 248, "cardiovascular_max")  # Redundant constraint, but included for completeness
    m.addConstr(0.73 * vitamin_c + 4.16 * vitamin_b7 <= 90, "digestive_max") # Redundant constraint, but included for completeness
    m.addConstr(0.37 * vitamin_c + 1.26 * vitamin_b7 <= 128, "kidney_max") # Redundant constraint, but included for completeness
    m.addConstr(3.56 * vitamin_c + 1.87 * vitamin_b7 <= 127, "energy_max") # Redundant constraint, but included for completeness


    # Optimize model
    m.optimize()

    # Print results
    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('Vitamin C: %g' % vitamin_c.x)
        print('Vitamin B7: %g' % vitamin_b7.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('Optimization problem is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)

except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
