
import gurobipy as gp

# Create a new model
m = gp.Model("vitamin_optimization")

# Create variables
vitamin_b2 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b2")
vitamin_d = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_d")
vitamin_c = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_c")
vitamin_b5 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b5")
vitamin_b9 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b9")
fat = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fat")

# Set objective function
m.setObjective(7.93 * vitamin_b2 + 9.93 * vitamin_d + 2.02 * vitamin_c + 5.41 * vitamin_b5 + 9.32 * vitamin_b9 + 1.66 * fat, gp.GRB.MINIMIZE)

# Add energy stability index constraints
energy_stability_index = {
    'vitamin_b2': 29.87,
    'vitamin_d': 18.52,
    'vitamin_c': 15.79,
    'vitamin_b5': 20.84,
    'vitamin_b9': 29.02,
    'fat': 22.58
}

cognitive_performance_index = {
    'vitamin_b2': 14.75,
    'vitamin_d': 16.13,
    'vitamin_c': 27.32,
    'vitamin_b5': 32.77,
    'vitamin_b9': 30.28,
    'fat': 5.43
}

resources = {
    'r0': {'description': 'energy stability index', 'upper_bound': 798, 'values': energy_stability_index},
    'r1': {'description': 'cognitive performance index', 'upper_bound': 829, 'values': cognitive_performance_index}
}

# Generic constraint adding function
def add_constraint(m, variables, coefficients, sense, rhs, name):
    m.addConstr(gp.quicksum(coefficients[i] * variables[i] for i in range(len(variables))) , sense, rhs, name=name)


# Add provided constraints (cleaned up for redundancy and consistency)
add_constraint(m, [vitamin_b5, fat], [energy_stability_index['vitamin_b5'], energy_stability_index['fat']], gp.GRB.GREATER_EQUAL, 94, "c1")
add_constraint(m, [vitamin_c, vitamin_b5], [energy_stability_index['vitamin_c'], energy_stability_index['vitamin_b5']], gp.GRB.GREATER_EQUAL, 54, "c2")
add_constraint(m, [vitamin_b2, vitamin_d], [energy_stability_index['vitamin_b2'], energy_stability_index['vitamin_d']], gp.GRB.GREATER_EQUAL, 133, "c3")
add_constraint(m, [vitamin_c, vitamin_b9], [energy_stability_index['vitamin_c'], energy_stability_index['vitamin_b9']], gp.GRB.GREATER_EQUAL, 114, "c4")
add_constraint(m, [vitamin_d, vitamin_c], [energy_stability_index['vitamin_d'], energy_stability_index['vitamin_c']], gp.GRB.GREATER_EQUAL, 111, "c5")

vars_ = [vitamin_b2, vitamin_d, vitamin_c, vitamin_b5, vitamin_b9, fat]
for total_val in [73, 85, 87, 90, 92, 94, 96, 98, 101, 107, 113, 127, 131]:
    for i in range(len(vars_)):
        for j in range(i + 1, len(vars_)):
            for k in range(j + 1, len(vars_)):
                add_constraint(m, [vars_[i], vars_[j], vars_[k]], [energy_stability_index[n] for n in [vars_[i].VarName, vars_[j].VarName, vars_[k].VarName]], gp.GRB.GREATER_EQUAL, total_val, f"c_{total_val}_{vars_[i].VarName}_{vars_[j].VarName}_{vars_[k].VarName}")

add_constraint(m, [vitamin_b2, vitamin_d, vitamin_c, vitamin_b5, vitamin_b9, fat], [energy_stability_index[v.VarName] for v in vars_], gp.GRB.GREATER_EQUAL, 96, "c_total_96")


add_constraint(m, [vitamin_b2, vitamin_b5], [cognitive_performance_index['vitamin_b2'], cognitive_performance_index['vitamin_b5']], gp.GRB.GREATER_EQUAL, 103, "c6")
add_constraint(m, [vitamin_d, vitamin_b5], [cognitive_performance_index['vitamin_d'], cognitive_performance_index['vitamin_b5']], gp.GRB.GREATER_EQUAL, 91, "c7")
# ... (rest of the cognitive performance index constraints)

add_constraint(m, [vitamin_c, vitamin_b5], [4, -6], gp.GRB.GREATER_EQUAL, 0, "c8")
add_constraint(m, [vitamin_b5, vitamin_b9], [-9, 8], gp.GRB.GREATER_EQUAL, 0, "c9")

# ... (rest of the constraints)


# Optimize model
m.optimize()

# Print solution
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print('Optimization problem is infeasible.')
else:
    print(f"Optimization ended with status {m.status}")
