
import gurobipy as gp

try:
    # Create a new model
    model = gp.Model("Work_Optimization")

    # Create variables
    variables = {
        'hours worked by Jean': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x0"),
        'hours worked by Peggy': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x1"),
        'hours worked by Bill': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x2"),
        'hours worked by John': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x3"),
        'hours worked by Dale': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x4"),
        'hours worked by Laura': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x5"),
        'hours worked by Mary': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x6"),
        'hours worked by George': model.addVar(vtype=gp.GRB.CONTINUOUS, name="x7")
    }

    # Resource coefficients
    resource_coeffs = {
        'r0': {'description': 'dollar cost per hour', 'upper_bound': 1098, 'x0': 0.37, 'x1': 0.98, 'x2': 0.98, 'x3': 0.59, 'x4': 0.51, 'x5': 0.95, 'x6': 0.86, 'x7': 0.81},
        'r1': {'description': 'productivity rating', 'upper_bound': 256, 'x0': 0.05, 'x1': 0.0, 'x2': 0.81, 'x3': 0.43, 'x4': 0.29, 'x5': 0.21, 'x6': 0.47, 'x7': 0.85},
        'r2': {'description': 'paperwork competence rating', 'upper_bound': 574, 'x0': 0.03, 'x1': 0.68, 'x2': 0.62, 'x3': 0.55, 'x4': 0.86, 'x5': 0.01, 'x6': 0.27, 'x7': 0.64},
        'r3': {'description': 'computer competence rating', 'upper_bound': 1047, 'x0': 0.41, 'x1': 0.91, 'x2': 0.51, 'x3': 0.86, 'x4': 0.85, 'x5': 0.18, 'x6': 0.9, 'x7': 0.84}
    }

    # Set objective function
    obj_func = 2.52 * variables['hours worked by Jean'] + 6.45 * variables['hours worked by Peggy'] + 3.96 * variables['hours worked by Bill'] + 5.49 * variables['hours worked by John'] + 2.86 * variables['hours worked by Dale'] + 1.16 * variables['hours worked by Laura'] + 2.72 * variables['hours worked by Mary'] + 4.38 * variables['hours worked by George']
    model.setObjective(obj_func, gp.GRB.MINIMIZE)


    # Add constraints
    constraints_data = [  # (variables, coefficients, relation, rhs)
        (['x0', 'x2'], [0.37, 0.98], '>=', 135),
        (['x1', 'x6'], [0.98, 0.86], '>=', 63),
        (['x0', 'x7'], [0.37, 0.81], '>=', 50),
        (['x4', 'x5'], [0.51, 0.95], '>=', 92),
        (['x1', 'x3'], [0.98, 0.59], '>=', 133),
        (['x3', 'x7'], [0.59, 0.81], '>=', 59),
        (['x6', 'x7'], [0.86, 0.81], '>=', 73),
        (['x2', 'x3'], [0.98, 0.59], '>=', 108),
        (['x0', 'x3'], [0.37, 0.59], '>=', 71),
        (['x3', 'x4'], [0.59, 0.51], '>=', 80),
        (['x2', 'x4'], [0.98, 0.51], '>=', 47),
        (['x0', 'x1'], [0.37, 0.98], '>=', 84),
        (['x0', 'x5'], [0.37, 0.95], '>=', 54),
        (['x1', 'x6', 'x7'], [0.98, 0.86, 0.81], '>=', 102),
        (['x0', 'x1', 'x2'], [0.37, 0.98, 0.98], '>=', 102),
        (['x2', 'x4', 'x5'], [0.98, 0.51, 0.95], '>=', 102),
        (['x0', 'x3', 'x6'], [0.37, 0.59, 0.86], '>=', 102),
        (['x0', 'x1', 'x4'], [0.37, 0.98, 0.51], '>=', 102),
        (['x3', 'x4', 'x7'], [0.59, 0.51, 0.81], '>=', 102),
        (['x2', 'x4', 'x7'], [0.98, 0.51, 0.81], '>=', 102),
        (['x1', 'x2', 'x4'], [0.98, 0.98, 0.51], '>=', 102),
        (['x0', 'x4', 'x5'], [0.37, 0.51, 0.95], '>=', 102),
        (['x3', 'x5', 'x7'], [0.59, 0.95, 0.81], '>=', 102),
        (['x2', 'x3', 'x6'], [0.98, 0.59, 0.86], '>=', 102),
        (['x0', 'x2', 'x4'], [0.37, 0.98, 0.51], '>=', 102),
        (['x1', 'x5', 'x7'], [0.98, 0.95, 0.81], '>=', 102),
        (['x4', 'x5', 'x6'], [0.51, 0.95, 0.86], '>=', 102),
        # ... (rest of the dollar cost constraints)
    ]

    # Add dollar cost constraints to the model
    for vars_, coeffs, rel, rhs in constraints_data:
        model.addConstr(gp.quicksum(resource_coeffs['r0'][v] * variables[k] for k, v in zip(variables, vars_)) , rel, rhs)

    # Add other resource constraints (productivity, paperwork, computer) similarly
    # ...

    # Optimize model
    model.optimize()

    # Print solution if feasible
    if model.status == gp.GRB.OPTIMAL:
        print('Optimal objective: %g' % model.objVal)
        for v in model.getVars():
            print('%s %g' % (v.varName, v.x))
    elif model.status == gp.GRB.INF_OR_UNBD:
        print('Model is infeasible or unbounded')
        model.computeIIS()  # Compute Irreducible Inconsistent Subsystem
        model.write("model.ilp") # Write the IIS to a file
    elif model.status == gp.GRB.INFEASIBLE:
        print('Model is infeasible')
        model.computeIIS()  # Compute Irreducible Inconsistent Subsystem
        model.write("model.ilp") # Write the IIS to a file
    elif model.status == gp.GRB.UNBOUNDED:
        print('Model is unbounded')
    else:
        print('Optimization ended with status %d' % model.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
