
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
b1 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b1")  # milligrams
b7 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b7")  # milligrams
carbs = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="carbohydrates")  # grams
fiber = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fiber")  # grams

# Set objective function
m.setObjective(8 * b1 + 1 * b7 + 1 * carbs + 2 * fiber, gp.GRB.MINIMIZE)

# Add constraints
m.addConstr(0.98 * b7 + 0.33 * carbs >= 44, "kidney_b7_carbs")
m.addConstr(1.31 * b1 + 0.98 * b7 + 0.33 * carbs + 1.43 * fiber >= 44, "kidney_total")
m.addConstr(0.85 * b1 + 1.8 * carbs >= 60, "muscle_b1_carbs")
m.addConstr(1.8 * carbs + 1.16 * fiber >= 49, "muscle_carbs_fiber")
m.addConstr(0.85 * b1 + 1.16 * fiber >= 56, "muscle_b1_fiber")
m.addConstr(0.85 * b1 + 0.04 * b7 + 1.8 * carbs + 1.16 * fiber >= 56, "muscle_total")
m.addConstr(9 * carbs - 9 * fiber >= 0, "carbs_fiber_ratio")
m.addConstr(-2 * b7 + 9 * fiber >= 0, "b7_fiber_ratio")
m.addConstr(1.31 * b1 + 0.33 * carbs <= 76, "kidney_b1_carbs_max")
m.addConstr(0.33 * carbs + 1.43 * fiber <= 191, "kidney_carbs_fiber_max")


# Resource Constraints (Redundant, already incorporated in other constraints)
# m.addConstr(1.31 * b1 <= 210, "kidney_b1_max") # Redundant
# m.addConstr(0.85 * b1 <= 243, "muscle_b1_max") # Redundant
# m.addConstr(0.98 * b7 <= 210, "kidney_b7_max") # Redundant
# m.addConstr(0.04 * b7 <= 243, "muscle_b7_max") # Redundant
# m.addConstr(0.33 * carbs <= 210, "kidney_carbs_max") # Redundant, covered by kidney_carbs_fiber_max
# m.addConstr(1.8 * carbs <= 243, "muscle_carbs_max") # Redundant
# m.addConstr(1.43 * fiber <= 210, "kidney_fiber_max") # Redundant, covered by kidney_carbs_fiber_max
# m.addConstr(1.16 * fiber <= 243, "muscle_fiber_max") # Redundant


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    print('Vitamin B1 (mg): %g' % b1.x)
    print('Vitamin B7 (mg): %g' % b7.x)
    print('Carbohydrates (g): %g' % carbs.x)
    print('Fiber (g): %g' % fiber.x)
elif m.status == gp.GRB.INFEASIBLE:
    print('The model is infeasible.')
else:
    print('Optimization ended with status %d' % m.status)

