
import gurobipy as gp

# Create a new model
m = gp.Model("Plant_Optimization")

# Create variables
plants = ['decorative cabbages', 'coleus', 'daisies', 'basil plants', 'aloe vera', 'chrysanthemums']
x = m.addVars(plants, vtype=gp.GRB.INTEGER, name=plants)

# Set objective function
m.setObjective(7 * x['decorative cabbages'] + 8 * x['coleus'] + 2 * x['daisies'] + 6 * x['basil plants'] + 9 * x['aloe vera'] + 6 * x['chrysanthemums'], gp.GRB.MAXIMIZE)

# Resource constraints
resources = {
    'r0': {'description': 'yield', 'upper_bound': 238, 'x0': 17, 'x1': 9, 'x2': 24, 'x3': 26, 'x4': 15, 'x5': 21},
    'r1': {'description': 'growth speed', 'upper_bound': 690, 'x0': 10, 'x1': 9, 'x2': 7, 'x3': 18, 'x4': 16, 'x5': 13},
    'r2': {'description': 'resilience index', 'upper_bound': 389, 'x0': 8, 'x1': 8, 'x2': 24, 'x3': 9, 'x4': 19, 'x5': 22},
    'r3': {'description': 'water need', 'upper_bound': 370, 'x0': 15, 'x1': 11, 'x2': 12, 'x3': 18, 'x4': 24, 'x5': 25}
}

for r_key, r_data in resources.items():
    m.addConstr(
        gp.quicksum(r_data[f'x{i}'] * x[plants[i]] for i in range(len(plants))) <= r_data['upper_bound'],
        name=r_data['description']
    )


# Additional constraints (growth speed, resilience, water need, yield combinations)
constraints = [
    (10*x['decorative cabbages'] + 13*x['chrysanthemums'] >= 40, "growth_speed_0_5"),
    (9*x['coleus'] + 16*x['aloe vera'] >= 103, "growth_speed_1_4"),
    # ... (All other constraints from the input)
]

for constraint, name in constraints:
    m.addConstr(constraint, name=name)

# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print("Optimal solution found:")
    for plant in plants:
        print(f"{plant}: {x[plant].x}")
    print(f"Objective value: {m.objVal}")
elif m.status == gp.GRB.INFEASIBLE:
    print("Model is infeasible.")
else:
    print(f"Optimization terminated with status {m.status}")

