
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
fiber = m.addVar(lb=0, name="fiber")
vitamin_b9 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b9")
protein = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="protein")

# Set objective function
m.setObjective(9.54 * fiber + 4.75 * vitamin_b9 + 3.24 * protein, gp.GRB.MINIMIZE)

# Add constraints
m.addConstr(1.04 * fiber + 0.49 * vitamin_b9 >= 117, "immune_constraint1")
m.addConstr(1.04 * fiber + 0.26 * protein >= 109, "immune_constraint2")
m.addConstr(1.04 * fiber + 0.49 * vitamin_b9 + 0.26 * protein >= 109, "immune_constraint3")
m.addConstr(2.48 * fiber + 0.8 * protein >= 93, "energy_constraint1")
m.addConstr(2.48 * fiber + 0.03 * vitamin_b9 >= 46, "energy_constraint2")
m.addConstr(2.48 * fiber + 0.03 * vitamin_b9 + 0.8 * protein >= 46, "energy_constraint3")
m.addConstr(0.44 * fiber + 2.45 * protein >= 85, "cognitive_constraint1")
m.addConstr(0.44 * fiber + 0.64 * vitamin_b9 >= 34, "cognitive_constraint2")
m.addConstr(0.44 * fiber + 0.64 * vitamin_b9 + 2.45 * protein >= 34, "cognitive_constraint3")
m.addConstr(6 * vitamin_b9 - 8 * protein >= 0, "constraint4")
m.addConstr(2 * fiber - 3 * vitamin_b9 >= 0, "constraint5")
m.addConstr(0.49 * vitamin_b9 + 0.26 * protein <= 148, "immune_constraint4")
m.addConstr(0.44 * fiber + 0.64 * vitamin_b9 <= 156, "cognitive_constraint4")


# Resource Constraints based on the provided dictionary
resources = {'r0': {'description': 'immune support index', 'upper_bound': 360, 'x0': 1.04, 'x1': 0.49, 'x2': 0.26}, 
            'r1': {'description': 'energy stability index', 'upper_bound': 343, 'x0': 2.48, 'x1': 0.03, 'x2': 0.8}, 
            'r2': {'description': 'cognitive performance index', 'upper_bound': 282, 'x0': 0.44, 'x1': 0.64, 'x2': 2.45}}

for r_key, r_data in resources.items():
    m.addConstr(r_data['x0'] * fiber + r_data['x1'] * vitamin_b9 + r_data['x2'] * protein <= r_data['upper_bound'], r_key)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print(f"Optimal objective value: {m.objVal}")
    print(f"Grams of fiber: {fiber.x}")
    print(f"Milligrams of vitamin B9: {vitamin_b9.x}")
    print(f"Grams of protein: {protein.x}")
elif m.status == gp.GRB.INFEASIBLE:
    print("The model is infeasible.")
else:
    print(f"Optimization ended with status {m.status}")

