
import gurobipy as gp
from gurobipy import GRB

# Create a new model
m = gp.Model("john_diet")

# Define the food options and their nutritional content
foods = {
    "rice": {"protein": 2, "carbs": 80, "fat": 1, "cost": 5},
    "beef": {"protein": 20, "carbs": 200, "fat": 16, "cost": 30},
}

# Define the minimum daily macronutrient requirements
requirements = {"protein": 50, "carbs": 1000, "fat": 40}

# Create decision variables for the number of servings of each food
servings = {}
for food in foods:
    servings[food] = m.addVar(vtype=GRB.CONTINUOUS, name=f"servings_{food}")


# Set the objective function: minimize total cost
m.setObjective(
    gp.quicksum(foods[food]["cost"] * servings[food] for food in foods), GRB.MINIMIZE
)

# Add constraints for minimum macronutrient requirements
for nutrient in requirements:
    m.addConstr(
        gp.quicksum(foods[food][nutrient] * servings[food] for food in foods)
        >= requirements[nutrient],
        name=f"{nutrient}_requirement",
    )
    
# Add non-negativity constraints
for food in foods:
    m.addConstr(servings[food] >= 0)


# Optimize the model
m.optimize()

# Print the results
if m.status == GRB.OPTIMAL:
    print("Optimal solution found:")
    for food in foods:
        print(f"Servings of {food}: {servings[food].x}")
    print(f"Total cost: ${m.objVal}")
elif m.status == GRB.INFEASIBLE:
    print("The model is infeasible. No solution exists that meets all requirements.")
else:
    print(f"Optimization terminated with status {m.status}")

