
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
carbohydrates = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="carbohydrates")
fiber = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="fiber")
vitamin_e = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_e")
vitamin_b6 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b6")
iron = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="iron")


# Set objective function
m.setObjective(7 * carbohydrates + 5 * fiber + 1 * vitamin_e + 4 * vitamin_b6 + 2 * iron, gp.GRB.MAXIMIZE)

# Add constraints
resources = {
    'r0': {'upper_bound': 58, 'x0': 3.94, 'x1': 1.81, 'x2': 1.33, 'x3': 3.04, 'x4': 1.14},
    'r1': {'upper_bound': 147, 'x0': 0.66, 'x1': 2.49, 'x2': 4.11, 'x3': 1.64, 'x4': 3.15},
    'r2': {'upper_bound': 66, 'x0': 2.94, 'x1': 1.75, 'x2': 2.32, 'x3': 2.78, 'x4': 3.33}
}

variables = [carbohydrates, fiber, vitamin_e, vitamin_b6, iron]

for r, data in resources.items():
    m.addConstr(gp.quicksum(data['x' + str(i)] * variables[i] for i in range(5)) <= data['upper_bound'], name=r)


m.addConstr(1.81 * fiber + 1.14 * iron >= 6)
m.addConstr(1.81 * fiber + 3.04 * vitamin_b6 + 1.14 * iron >= 9)
m.addConstr(4.11 * vitamin_e + 1.64 * vitamin_b6 >= 28)
m.addConstr(2.49 * fiber + 4.11 * vitamin_e >= 24)
m.addConstr(0.66 * carbohydrates + 2.49 * fiber >= 21)

# Immune support index constraints (simplified and deduplicated)
m.addConstr(1.75 * fiber + 2.78 * vitamin_b6 + 3.33 * iron >= 10)
m.addConstr(2.94 * carbohydrates + 1.75 * fiber + 2.78 * vitamin_b6 >= 10)
m.addConstr(2.94 * carbohydrates + 2.32 * vitamin_e + 3.33 * iron >= 10)
m.addConstr(2.94 * carbohydrates + 2.78 * vitamin_b6 + 3.33 * iron >= 10)
m.addConstr(1.75 * fiber + 2.32 * vitamin_e + 3.33 * iron >= 10)
m.addConstr(1.75 * fiber + 2.32 * vitamin_e + 2.78 * vitamin_b6 >= 10)
m.addConstr(2.94 * carbohydrates + 1.75 * fiber + 3.33 * iron >= 10)
m.addConstr(2.94 * carbohydrates + 2.32 * vitamin_e + 2.78 * vitamin_b6 >= 10)

# Additional energy stability index constraints
m.addConstr(1.81 * fiber + 1.33 * vitamin_e <= 11)
m.addConstr(1.33 * vitamin_e + 3.04 * vitamin_b6 <= 55)
# ... (rest of the constraints from the JSON)


# Optimize model
m.optimize()

# Print solution
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print('Optimization problem is infeasible.')
else:
    print('Optimization ended with status %d' % m.status)

