
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    x0 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="x0") # milligrams of vitamin B12
    x1 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="x1") # milligrams of vitamin B7


    # Set objective function
    m.setObjective(9.14 * x0 + 5.53 * x1, gp.GRB.MAXIMIZE)

    # Add constraints
    m.addConstr(11 * x0 + 10 * x1 >= 23, "c0") # cardiovascular support index minimum
    m.addConstr(12 * x0 + 3 * x1 >= 16, "c1") # energy stability index minimum
    m.addConstr(4 * x0 - 3 * x1 >= 0, "c2") # constraint on B12 and B7
    m.addConstr(11 * x0 + 10 * x1 <= 49, "c3") # cardiovascular support index maximum
    m.addConstr(12 * x0 + 3 * x1 <= 50, "c4") # energy stability index maximum


    # Optimize model
    m.optimize()

    # Print results
    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('x0: %g' % x0.x)
        print('x1: %g' % x1.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('Optimization problem is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')

