
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
protein = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="protein")
vitamin_b5 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b5")
fat = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fat")
vitamin_b7 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b7")
vitamin_b4 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b4")
vitamin_e = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_e")
vitamin_c = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_c")


# Set objective function
m.setObjective(6.57 * protein + 1.58 * vitamin_b5 + 2.49 * fat + 7.78 * vitamin_b7 + 4.18 * vitamin_b4 + 4.85 * vitamin_e + 5.57 * vitamin_c, gp.GRB.MINIMIZE)

# Add cardiovascular support index constraints
cardiovascular_support = {
    'protein': 3,
    'vitamin_b5': 5,
    'fat': 2,
    'vitamin_b7': 6,
    'vitamin_b4': 7,
    'vitamin_e': 4,
    'vitamin_c': 11
}

muscle_growth = {
    'protein': 8,
    'vitamin_b5': 5,
    'fat': 5,
    'vitamin_b7': 4,
    'vitamin_b4': 8,
    'vitamin_e': 6,
    'vitamin_c': 7
}

resource_constraints = {
    'r0': {'upper_bound': 200, 'description': 'cardiovascular support index', **cardiovascular_support},
    'r1': {'upper_bound': 181, 'description': 'muscle growth index', **muscle_growth}
}


variables = {
    'protein': protein,
    'vitamin_b5': vitamin_b5,
    'fat': fat,
    'vitamin_b7': vitamin_b7,
    'vitamin_b4': vitamin_b4,
    'vitamin_e': vitamin_e,
    'vitamin_c': vitamin_c
}

# Generic constraint adder for single, double, and triple combinations
def add_constraint(m, resource, var_names, lower_bound=None, upper_bound=None):
    coefficients = [resource_constraints[resource][var_name] for var_name in var_names]
    variables_ = [variables[var_name] for var_name in var_names]
    expression = gp.LinExpr(coefficients, variables_)
    if lower_bound is not None:
        m.addConstr(expression >= lower_bound)
    if upper_bound is not None:
        m.addConstr(expression <= upper_bound)

# Add constraints from JSON (and other constraints from the problem description)
for constraint in json_data['constraints']:
    parts = constraint.split(" ")
    lower_bound = None
    upper_bound = None
    if ">=" in constraint:
        lower_bound = float(parts[-1])
    elif "<=" in constraint:
        upper_bound = float(parts[-1])
    var_names = []
    for i in range(0, len(parts) - 2, 2):
        var_names.append(json_data['sym_variables'][int(parts[i][1:])][1].replace("number of ", "").replace("amount of ", "").replace("quantity of ", "").strip())
    add_constraint(m, 'r0', var_names, lower_bound, upper_bound)

for constraint in json_data['constraints'][772:]:
    parts = constraint.split(" ")
    lower_bound = None
    upper_bound = None
    if ">=" in constraint:
        lower_bound = float(parts[-1])
    elif "<=" in constraint:
        upper_bound = float(parts[-1])
    var_names = []
    for i in range(0, len(parts) - 2, 2):
        var_names.append(json_data['sym_variables'][int(parts[i][1:])][1].replace("number of ", "").replace("amount of ", "").replace("quantity of ", "").strip())
    add_constraint(m, 'r1', var_names, lower_bound, upper_bound)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print('Optimization was infeasible.')
else:
    print('Optimization ended with status %d' % m.status)
