
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("nutrition_optimization")

    # Create variables
    vitamin_a = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_a")
    vitamin_b5 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b5")
    protein = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="protein")
    fat = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fat")

    # Set objective function
    m.setObjective(9.11 * vitamin_a + 2.55 * vitamin_b5 + 5.21 * protein + 9.45 * fat, gp.GRB.MAXIMIZE)

    # Add constraints
    m.addConstr(8 * vitamin_a + 7 * vitamin_b5 + 15 * fat >= 31, "digestive_support_min")
    m.addConstr(8 * vitamin_a + 7 * vitamin_b5 <= 89, "digestive_support_ab_max")
    m.addConstr(7 * vitamin_b5 + 2 * protein <= 111, "digestive_support_bp_max")
    m.addConstr(8 * vitamin_a + 15 * fat <= 95, "digestive_support_af_max")
    m.addConstr(7 * vitamin_b5 + 15 * fat <= 94, "digestive_support_bf_max")
    m.addConstr(8 * vitamin_a + 7 * vitamin_b5 + 15 * fat <= 142, "digestive_support_abf_max")
    m.addConstr(8 * vitamin_a + 7 * vitamin_b5 + 2 * protein + 15 * fat <= 142, "digestive_support_total_max")


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('Vitamin A: %g' % vitamin_a.x)
        print('Vitamin B5: %g' % vitamin_b5.x)
        print('Protein: %g' % protein.x)
        print('Fat: %g' % fat.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
