
import gurobipy as gp
from gurobipy import GRB

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    x0 = m.addVar(vtype=GRB.INTEGER, name="vitamin_A")  # milligrams of vitamin A (integer)
    x1 = m.addVar(vtype=GRB.CONTINUOUS, name="vitamin_B5") # milligrams of vitamin B5

    # Set objective function
    m.setObjective(1.66*x0**2 + 2.23*x1**2 + 5.28*x1, GRB.MINIMIZE)

    # Add constraints
    m.addConstr(20*x0 + 1*x1 >= 32, "immune_support_min")
    m.addConstr(20*x0 + 1*x1 <= 54, "immune_support_max")
    m.addConstr(13*x0 + 5*x1 >= 11, "cardio_support_min")
    m.addConstr(13*x0 + 5*x1 <= 43, "cardio_support_max")
    m.addConstr(5*x0**2 - 4*x1**2 >= 0, "quadratic_constraint")


    # Optimize model
    m.optimize()

    # Print solution
    if m.status == GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('Vitamin A: %g' % x0.x)
        print('Vitamin B5: %g' % x1.x)
    elif m.status == GRB.INFEASIBLE:
        print('Model is infeasible')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')

