
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
x = {}
var_names = [
    "milligrams of vitamin B6",
    "milligrams of vitamin E",
    "milligrams of vitamin B9",
    "grams of fiber",
    "milligrams of iron",
    "milligrams of vitamin K",
    "milligrams of vitamin B5",
]
for i in range(len(var_names)):
    if var_names[i] == "milligrams of vitamin B9":
        x[i] = m.addVar(vtype=gp.GRB.INTEGER, name=var_names[i])
    else:
        x[i] = m.addVar(name=var_names[i])


# Set objective function
m.setObjective(
    9.43 * x[0]
    + 6.8 * x[1]
    + 9.05 * x[2]
    + 6.01 * x[3]
    + 4.84 * x[4]
    + 2.36 * x[5]
    + 7.56 * x[6],
    gp.GRB.MINIMIZE,
)

# Resource data
resources = {
    'r0': {'description': 'energy stability index', 'upper_bound': 518, 'x0': 9, 'x1': 10, 'x2': 8, 'x3': 3, 'x4': 8, 'x5': 6, 'x6': 11},
    'r1': {'description': 'immune support index', 'upper_bound': 588, 'x0': 3, 'x1': 17, 'x2': 10, 'x3': 11, 'x4': 14, 'x5': 4, 'x6': 7},
    'r2': {'description': 'cognitive performance index', 'upper_bound': 227, 'x0': 4, 'x1': 5, 'x2': 16, 'x3': 14, 'x4': 12, 'x5': 9, 'x6': 10}
}

# Add resource constraints
for r_key, r_data in resources.items():
    m.addConstr(gp.quicksum(r_data[f'x{i}'] * x[i] for i in range(7)) <= r_data['upper_bound'], name=r_data['description'] + '_upper_bound')


# Add other constraints from the problem description
constraints = [
    (9 * x[4] + 6 * x[5], ">=", 44),
    # ... (all other constraints)
]

for expr, sense, rhs in constraints:
    if sense == ">=":
        m.addConstr(expr >= rhs)
    elif sense == "<=":
        m.addConstr(expr <= rhs)
    elif sense == "=":
        m.addConstr(expr == rhs)

# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print("Optimal solution found:")
    for i in range(len(var_names)):
        print(f"{var_names[i]}: {x[i].x}")
    print(f"Objective value: {m.objVal}")
elif m.status == gp.GRB.INFEASIBLE:
    print("Model is infeasible.")
else:
    print(f"Optimization terminated with status {m.status}")

