
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("optimization_problem")

    # Create variables
    x = {}
    var_names = ['system administrators', 'security engineers', 'security onions', 
                 'Mbps bandwidth allocated to monitoring', 'automatic alerts', 'SOC operators']
    for i in range(len(var_names)):
        x[i] = m.addVar(vtype=gp.GRB.INTEGER, name=var_names[i])

    # Set objective function
    obj_coeffs = [8.63, 6.26, 6.28, 5.24, 5.69, 3.69]
    m.setObjective(gp.quicksum(obj_coeffs[i] * x[i] for i in range(len(var_names))), gp.GRB.MINIMIZE)


    # Resource constraints
    resources = {'r0': {'upper_bound': 92, 'coeffs': [10, 10, 6, 7, 3, 4]},
                 'r1': {'upper_bound': 430, 'coeffs': [11, 1, 7, 7, 1, 15]},
                 'r2': {'upper_bound': 114, 'coeffs': [16, 6, 8, 11, 17, 3]},
                 'r3': {'upper_bound': 335, 'coeffs': [13, 4, 14, 13, 4, 13]}}

    for resource, data in resources.items():
        m.addConstr(gp.quicksum(data['coeffs'][i] * x[i] for i in range(len(var_names))) <= data['upper_bound'], name=resource)


    # Add other constraints (example subset - add all constraints similarly)
    m.addConstr(10 * x[0] + 7 * x[3] >= 9)
    m.addConstr(6 * x[2] + 3 * x[4] >= 15)
    m.addConstr(x[1] + x[4] >= 64)
    m.addConstr(16 * x[0] + 3 * x[5] >= 19)
    m.addConstr(6 * x[1] + 8 * x[2] >= 10)


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        solution = {}
        for i in range(len(var_names)):
            solution[var_names[i]] = x[i].x
        print(solution)
        print('Obj: %g' % m.objVal)
    elif m.status == gp.GRB.INFEASIBLE:
        print("The model is infeasible.")
    else:
        print("Optimization ended with status:", m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
