
import gurobipy as gp
from gurobipy import GRB

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    x0 = m.addVar(lb=0, vtype=GRB.CONTINUOUS, name="vitamin_B1") # milligrams of vitamin B1
    x1 = m.addVar(lb=0, vtype=GRB.INTEGER, name="potassium") # milligrams of potassium


    # Set objective function
    m.setObjective(1*x0 + 6*x1, GRB.MINIMIZE)

    # Add constraints
    m.addConstr(9*x0 + 2*x1 >= 9, "c0") # cognitive performance index minimum
    m.addConstr(2*x0 + 14*x1 >= 16, "c1") # immune support index minimum
    m.addConstr(-10*x0 + 4*x1 >= 0, "c2") # constraint relating B1 and potassium
    m.addConstr(9*x0 + 2*x1 <= 19, "c3") # cognitive performance index maximum
    m.addConstr(2*x0 + 14*x1 <= 31, "c4") # immune support index maximum


    # Optimize model
    m.optimize()

    # Print solution
    if m.status == GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('Vitamin B1: %g' % x0.x)
        print('Potassium: %g' % x1.x)
    elif m.status == GRB.INFEASIBLE:
        print('Model is infeasible')
    else:
        print('Optimization ended with status %d' % m.status)

except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')

