
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    vitamin_c = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_c")
    fiber = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fiber")
    vitamin_b3 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b3")


    # Set objective function
    m.setObjective(6*vitamin_c + 6*fiber + 2*vitamin_b3, gp.GRB.MINIMIZE)

    # Add constraints
    m.addConstr(4*vitamin_c + 5*fiber >= 14, "c1")
    m.addConstr(5*fiber + vitamin_b3 >= 7, "c2")
    m.addConstr(4*vitamin_c + 5*fiber + vitamin_b3 >= 17, "c3")
    m.addConstr(-7*fiber + 7*vitamin_b3 >= 0, "c4")
    m.addConstr(-5*vitamin_c + vitamin_b3 >= 0, "c5")
    m.addConstr(-vitamin_c + 10*fiber >= 0, "c6")
    m.addConstr(5*fiber + vitamin_b3 <= 26, "c7")


    # Optimize model
    m.optimize()

    # Print solution
    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('vitamin_c:', vitamin_c.x)
        print('fiber:', fiber.x)
        print('vitamin_b3:', vitamin_b3.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
