
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
fat = m.addVar(vtype=gp.GRB.INTEGER, name="fat")
vitamin_b1 = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_b1")
vitamin_b4 = m.addVar(name="vitamin_b4")
fiber = m.addVar(vtype=gp.GRB.INTEGER, name="fiber")
iron = m.addVar(vtype=gp.GRB.INTEGER, name="iron")
vitamin_a = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_a")
calcium = m.addVar(vtype=gp.GRB.INTEGER, name="calcium")


# Set objective function
obj = 6.18*fat**2 + 6.53*fat*vitamin_b1 + 9.57*fat*fiber + 3.03*fat*vitamin_a + 9.78*vitamin_b1**2 + 3.19*vitamin_b1*vitamin_b4 + 2.03*vitamin_b1*iron + 3.66*vitamin_b1*calcium + 3.46*vitamin_b4*fiber + 4.36*vitamin_b4*iron + 3.22*vitamin_b4*vitamin_a + 7.62*vitamin_b4*calcium + 2.57*fiber**2 + 7.77*fiber*iron + 5.5*fiber*calcium + 5.11*iron*vitamin_a + 4.68*iron*calcium + 2.22*vitamin_a**2 + 1.61*vitamin_a*calcium + 9.0*calcium**2 + 6.81*fat + 4.04*vitamin_b4

m.setObjective(obj, gp.GRB.MAXIMIZE)

# Add constraints
resource_constraints = {
    'r0': {'upper_bound': 1197, 'coefficients': {'fat': 29, 'vitamin_b1': 9, 'vitamin_b4': 24, 'fiber': 16, 'iron': 4, 'vitamin_a': 9, 'calcium': 24}},
    'r1': {'upper_bound': 1042, 'coefficients': {'fat': 23, 'vitamin_b1': 11, 'vitamin_b4': 26, 'fiber': 15, 'iron': 10, 'vitamin_a': 9, 'calcium': 26}}
}

for resource, data in resource_constraints.items():
    m.addConstr(
        gp.quicksum(data['coefficients'][var_name] * vars()[var_name] for var_name in data['coefficients']) <= data['upper_bound'], name=resource
    )


m.addConstr(24 * vitamin_b4 + 4 * iron >= 150, "kidney_b4_iron")
m.addConstr(9 * vitamin_b1 + 9 * vitamin_a >= 117, "kidney_b1_a")
m.addConstr(29 * fat + 9 * vitamin_a >= 91, "kidney_fat_a")
m.addConstr(26 * vitamin_b4 + 15 * fiber >= 79, "cardio_b4_fiber")
m.addConstr(15 * fiber + 9 * vitamin_a >= 71, "cardio_fiber_a")
m.addConstr(11 * vitamin_b1**2 + 9 * vitamin_a**2 >= 111, "cardio_b1sq_asq")
m.addConstr(10 * iron + 9 * vitamin_a + 26 * calcium >= 90, "cardio_iron_a_calcium")



# Optimize model
m.optimize()

# Print solution
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print("Model is infeasible")
else:
    print("Optimization ended with status %d" % m.status)

