
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("nutrition_optimization")

    # Create variables
    vitamin_k = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_k")
    zinc = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="zinc")
    vitamin_b12 = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="vitamin_b12")

    # Set objective function
    m.setObjective(2*vitamin_k + 2*zinc + 5*vitamin_b12, gp.GRB.MINIMIZE)

    # Add constraints
    m.addConstr(6*vitamin_k + 8*zinc >= 10, "muscle_growth_1")
    m.addConstr(6*vitamin_k + 9*vitamin_b12 >= 12, "muscle_growth_2")
    m.addConstr(6*vitamin_k + 8*zinc + 9*vitamin_b12 >= 12, "muscle_growth_3")
    
    m.addConstr(6*vitamin_k + 3*zinc >= 25, "kidney_support_1")
    m.addConstr(6*vitamin_k + 8*vitamin_b12 >= 33, "kidney_support_2")
    m.addConstr(3*zinc + 8*vitamin_b12 >= 29, "kidney_support_3")
    m.addConstr(6*vitamin_k + 3*zinc + 8*vitamin_b12 >= 29, "kidney_support_4")
    
    m.addConstr(7*zinc + 9*vitamin_b12 >= 12, "cognitive_performance_1")
    m.addConstr(4*vitamin_k + 9*vitamin_b12 >= 10, "cognitive_performance_2")
    m.addConstr(4*vitamin_k + 7*zinc >= 13, "cognitive_performance_3")
    m.addConstr(4*vitamin_k + 7*zinc + 9*vitamin_b12 >= 12, "cognitive_performance_4")
    m.addConstr(4*vitamin_k + 7*zinc + 9*vitamin_b12 >= 12, "cognitive_performance_5")
    
    m.addConstr(-4*vitamin_k + 2*zinc >= 0, "constraint_1")
    m.addConstr(-9*vitamin_k + 3*vitamin_b12 >= 0, "constraint_2")
    m.addConstr(4*vitamin_k + 9*vitamin_b12 <= 45, "cognitive_performance_upper_bound_1")
    m.addConstr(4*vitamin_k + 7*zinc <= 58, "cognitive_performance_upper_bound_2")

    m.addConstr(6*vitamin_k <= 67, "muscle_growth_upper_bound")
    m.addConstr(3*zinc <= 106, "kidney_support_upper_bound")
    m.addConstr(9*vitamin_b12 <= 72, "cognitive_performance_upper_bound")


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('vitamin_k:', vitamin_k.x)
        print('zinc:', zinc.x)
        print('vitamin_b12:', vitamin_b12.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
