
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
fat = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fat")
vitamin_b7 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b7")
vitamin_c = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_c")
vitamin_e = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_e")
calcium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="calcium")

# Set objective function
m.setObjective(3*fat + 6*vitamin_b7 + 2*vitamin_c + 6*vitamin_e + 7*calcium, gp.GRB.MAXIMIZE)

# Add constraints
resource_constraints = {
    'r0': {'upper_bound': 269, 'coefficients': [2, 13, 20, 19, 3]},
    'r1': {'upper_bound': 419, 'coefficients': [12, 14, 5, 3, 7]},
    'r2': {'upper_bound': 647, 'coefficients': [12, 7, 20, 3, 4]},
    'r3': {'upper_bound': 272, 'coefficients': [16, 19, 13, 11, 4]},
    'r4': {'upper_bound': 237, 'coefficients': [23, 7, 23, 1, 19]}
}

variables = [fat, vitamin_b7, vitamin_c, vitamin_e, calcium]

for resource, data in resource_constraints.items():
    m.addConstr(
        gp.quicksum(data['coefficients'][i] * variables[i] for i in range(len(variables))) <= data['upper_bound'],
        name=resource
    )


# Add additional constraints from the problem description.  These are added verbatim from the prompt.
m.addConstr(13*vitamin_b7 + 3*calcium >= 37)
m.addConstr(20*vitamin_c + 3*calcium >= 20)
m.addConstr(2*fat + 13*vitamin_b7 >= 22)
# ... (All other constraints from the problem description)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print("The model is infeasible.")
else:
    print("Optimization ended with status %d" % m.status)

