
import gurobipy as gp

# Create a new model
m = gp.Model("nutrient_optimization")

# Create variables
vitamin_b6 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b6")
iron = m.addVar(lb=0, vtype=gp.GRB.INTEGER, name="iron")
vitamin_b5 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b5")
vitamin_b9 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b9")
magnesium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="magnesium")


# Set objective function
m.setObjective(7.6 * vitamin_b6 + 3.87 * iron + 2.34 * vitamin_b5 + 6.84 * vitamin_b9 + 2.44 * magnesium, gp.GRB.MAXIMIZE)

# Add constraints
resource_constraints = {
    'r0': {'upper_bound': 89, 'coeffs': [1, 2, 1, 3, 1]},
    'r1': {'upper_bound': 146, 'coeffs': [5, 11, 7, 11, 3]},
    'r2': {'upper_bound': 95, 'coeffs': [9, 1, 1, 2, 9]},
    'r3': {'upper_bound': 104, 'coeffs': [10, 4, 8, 5, 5]},
    'r4': {'upper_bound': 95, 'coeffs': [3, 1, 11, 11, 4]}
}

for r, data in resource_constraints.items():
    m.addConstr(
        data['coeffs'][0] * vitamin_b6 + data['coeffs'][1] * iron + data['coeffs'][2] * vitamin_b5 + data['coeffs'][3] * vitamin_b9 + data['coeffs'][4] * magnesium <= data['upper_bound'], name=r
    )


constraints_json = {
  "constraints": [
    "x0 + 2*x1 + x2 >= 16",
    "2*x1 + x2 + x4 >= 16",
    "x2 + 3*x3 + x4 >= 16",
    "x0 + 3*x3 + x4 >= 16",
    "x0 + 2*x1 + 3*x3 >= 16",
    "7*x2 + 11*x3 >= 25",
    "7*x2 + 11*x3 + 3*x4 >= 29",
    "11*x1 + 7*x2 + 3*x4 >= 29",
    "5*x0 + 7*x2 + 3*x4 >= 29",
    "11*x1 + 7*x2 + 11*x3 >= 29",
    "5*x0 + 11*x3 + 3*x4 >= 29",
    "5*x0 + 11*x1 + 3*x4 >= 29",
    "9*x0 + x2 >= 16",
    "x1 + 2*x3 >= 19",
    "4*x1 + 5*x4 >= 12",
    "10*x0 + 5*x4 >= 6",
    "5*x3 + 5*x4 >= 12",
    "10*x0 + 4*x1 >= 12",
    "10*x0 + 8*x2 >= 12",
    "8*x2 + 5*x3 >= 6",
    "8*x2 + 5*x4 >= 11",
    "11*x3 + 4*x4 >= 17",
    "x1 + 11*x3 >= 6",
    "3*x0 + 11*x3 >= 11",
    "3*x0 + 11*x2 >= 6",
    "x1 + 11*x2 >= 13",
    "x1 + 4*x4 >= 16",
    "3*x0 + x1 >= 6",
    "8*x1 - 8*x4 >= 0",
    "-6*x1 + 8*x2 >= 0",
    "2*x1 + x2 <= 84",
    "x0 + x4 <= 67",
    "2*x1 + x4 <= 72",
    "x2 + x4 <= 33",
    "x2 + 3*x3 <= 63",
    "x0 + 3*x3 <= 42",
    "2*x1 + x2 + x4 <= 89",
    "x0 + 3*x3 + x4 <= 57",
    "x0 + x2 + x4 <= 71",
    "x0 + x1 + x2 + x3 + x4 <= 71",
    "7*x2 + 3*x4 <= 59",
    "11*x1 + 7*x2 <= 136",
    "7*x2 + 11*x3 <= 131",
    "11*x3 + 3*x4 <= 43",
    "5*x0 + 3*x4 <= 127",
    "5*x0 + 11*x1 <= 104",
    "11*x1 + 3*x4 <= 118",
    "5*x0 + 7*x2 + 3*x4 <= 37",
    "11*x1 + 7*x2 + 11*x3 <= 118",
    "5*x0 + 11*x1 + 7*x2 + 11*x3 + 3*x4 <= 118",
    "2*x3 + 9*x4 <= 26",
    "9*x0 + 9*x4 <= 51",
    "x1 + 9*x4 <= 25",
    "x1 + x2 + 2*x3 <= 47",
    "x2 + 2*x3 + 9*x4 <= 36",
    "9*x0 + 2*x3 + 9*x4 <= 56",
    "9*x0 + x1 + x2 + 2*x3 + 9*x4 <= 56",
    "10*x0 + 4*x1 <= 70",
    "4*x1 + 5*x3 + 5*x4 <= 24",
    "10*x0 + 4*x1 + 5*x4 <= 98",
    "4*x1 + 8*x2 + 5*x4 <= 41",
    "10*x0 + 4*x1 + 5*x3 <= 42",
    "10*x0 + 4*x1 + 8*x2 + 5*x3 + 5*x4 <= 42",
    "x1 + 4*x4 <= 20",
    "3*x0 + 11*x2 <= 51",
    "3*x0 + 11*x3 <= 38",
    "x1 + 11*x2 <= 73",
    "x1 + 11*x2 + 11*x3 <= 23",
    "3*x0 + x1 + 11*x2 + 11*x3 + 4*x4 <= 23"
  ]
}


variables = [vitamin_b6, iron, vitamin_b5, vitamin_b9, magnesium]

for constraint_str in constraints_json['constraints']:
    lhs, rhs_str = constraint_str.split("<=")
    rhs = float(rhs_str)
    
    if ">=" in lhs:
        lhs, rhs_str = lhs.split(">=")
        rhs = float(rhs_str)
        coeffs = []
        for i, var in enumerate(variables):
            coeff_str = f"{i+1}*x{i}"
            coeff = 0
            if coeff_str in lhs:
                coeff = lhs.split(coeff_str)[0]
                if coeff == "":
                    coeff = 1
                elif coeff == "-":
                    coeff = -1
                else:
                    coeff = int(coeff)
            coeffs.append(coeff)
        m.addConstr(sum(c * v for c, v in zip(coeffs, variables)) >= rhs)
    else:
        coeffs = []
        for i, var in enumerate(variables):
            coeff_str = f"{i+1}*x{i}"
            coeff = 0
            if coeff_str in lhs:
                coeff = lhs.split(coeff_str)[0]
                if coeff == "":
                    coeff = 1
                elif coeff == "-":
                    coeff = -1
                else:
                    coeff = int(coeff)
            coeffs.append(coeff)
        m.addConstr(sum(c * v for c, v in zip(coeffs, variables)) <= rhs)


# Optimize model
m.optimize()

# Print results
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print("Model is infeasible")
else:
    print(f"Optimization ended with status {m.status}")

