
import gurobipy as gp

# Create a new model
m = gp.Model("food_optimization")

# Create variables
foods = ['hamburgers', 'bananas', 'granola bars', 'peanutbutter sandwiches', 'sashimi', 'chicken thighs']
x = m.addVars(foods, lb=0.0, name="x")


# Set objective function
m.setObjective(1.56 * x['hamburgers'] + 9.35 * x['bananas'] + 9.34 * x['granola bars'] + 2.73 * x['peanutbutter sandwiches'] + 7.13 * x['sashimi'] + 8.59 * x['chicken thighs'], gp.GRB.MINIMIZE)

# Add protein constraints
protein = {'hamburgers': 2, 'bananas': 11, 'granola bars': 4, 'peanutbutter sandwiches': 3, 'sashimi': 11, 'chicken thighs': 7}
m.addConstr(gp.quicksum(protein[f] * x[f] for f in foods) <= 213, "max_protein")


# Add iron constraints
iron = {'hamburgers': 6, 'bananas': 4, 'granola bars': 3, 'peanutbutter sandwiches': 2, 'sashimi': 9, 'chicken thighs': 11}
m.addConstr(gp.quicksum(iron[f] * x[f] for f in foods) <= 165, "max_iron")


# Add other constraints (provided in the prompt) - simplified and deduplicated
constraints = [
    (2*x['hamburgers'] + 3*x['peanutbutter sandwiches'], '>=', 27),
    (11*x['bananas'] + 7*x['chicken thighs'], '>=', 33),
    (11*x['bananas'] + 3*x['peanutbutter sandwiches'], '>=', 31),
    (4*x['granola bars'] + 7*x['chicken thighs'], '>=', 23),
    (2*x['hamburgers'] + 11*x['bananas'] + 7*x['chicken thighs'], '>=', 34),
    (2*x['hamburgers'] + 3*x['peanutbutter sandwiches'] + 7*x['chicken thighs'], '>=', 34),
    (11*x['bananas'] + 3*x['peanutbutter sandwiches'] + 11*x['sashimi'], '>=', 34),
    (3*x['peanutbutter sandwiches'] + 11*x['sashimi'] + 7*x['chicken thighs'], '>=', 34),
    (2*x['hamburgers'] + 4*x['granola bars'] + 11*x['sashimi'], '>=', 34),
    (x['granola bars'] - 5*x['chicken thighs'], '>=', 0),
    (5*x['bananas'] - 3*x['peanutbutter sandwiches'], '>=', 0),
    (-7*x['sashimi'] + 3*x['chicken thighs'], '>=', 0),
    (2*x['hamburgers'] + 11*x['sashimi'] + 7*x['chicken thighs'], '<=', 112),
    (2*x['hamburgers'] + 11*x['bananas'] + 4*x['granola bars'], '<=', 42)
]

for expr, sense, rhs in constraints:
    m.addConstr(expr, sense, rhs)


# Optimize model
m.optimize()

# Print solution or infeasibility status
if m.status == gp.GRB.OPTIMAL:
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
    print('Obj: %g' % m.objVal)
elif m.status == gp.GRB.INFEASIBLE:
    print("The model is infeasible.")
else:
    print(f"Optimization terminated with status {m.status}")

