
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("vitamin_optimization")

    # Create variables
    vitamin_e = m.addVar(lb=0, name="vitamin_e")  # Non-negative
    vitamin_k = m.addVar(lb=0, name="vitamin_k")  # Non-negative


    # Set objective function
    obj = 4.16*vitamin_e**2 + 8.88*vitamin_e*vitamin_k + 8.4*vitamin_k**2 + 1.6*vitamin_e + 8.51*vitamin_k
    m.setObjective(obj, gp.GRB.MAXIMIZE)

    # Add constraints
    m.addConstr(12*vitamin_e + 5*vitamin_k >= 15, "muscle_growth_min")
    m.addConstr(8*vitamin_e + 10*vitamin_k >= 26, "cardio_min")
    m.addConstr(2*vitamin_e + 12*vitamin_k >= 9, "immune_min")
    m.addConstr(5*vitamin_e**2 + 8*vitamin_k**2 >= 33, "cognitive_min")
    m.addConstr(-vitamin_e**2 + vitamin_k**2 >= 0, "e_k_relation")

    m.addConstr(12*vitamin_e + 5*vitamin_k <= 34, "muscle_growth_max")
    m.addConstr(8*vitamin_e + 10*vitamin_k <= 45, "cardio_max")
    m.addConstr(2*vitamin_e + 12*vitamin_k <= 27, "immune_max")
    m.addConstr(5*vitamin_e + 8*vitamin_k <= 72, "cognitive_max")


    # Optimize model
    m.optimize()

    # Print results
    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('Vitamin E: %g' % vitamin_e.x)
        print('Vitamin K: %g' % vitamin_k.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('Optimization problem is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
