
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("kidney_support_optimization")

    # Create variables
    calcium = m.addVar(vtype=gp.GRB.CONTINUOUS, name="calcium")
    vitamin_b6 = m.addVar(vtype=gp.GRB.CONTINUOUS, name="vitamin_b6")
    magnesium = m.addVar(vtype=gp.GRB.INTEGER, name="magnesium")
    fiber = m.addVar(vtype=gp.GRB.INTEGER, name="fiber")

    # Set objective function
    m.setObjective(5.58 * calcium + 8.21 * vitamin_b6 + 1.51 * magnesium + 6.53 * fiber, gp.GRB.MAXIMIZE)

    # Add constraints
    kidney_support_index = 18 * calcium + 12 * vitamin_b6 + 20 * magnesium + 26 * fiber
    m.addConstr(kidney_support_index <= 445, "kidney_support_limit")
    m.addConstr(18 * calcium + 26 * fiber >= 58, "calcium_fiber_min")
    m.addConstr(18 * calcium + 20 * magnesium <= 246, "calcium_magnesium_max")
    m.addConstr(18 * calcium + 12 * vitamin_b6 <= 219, "calcium_vitamin_b6_max")
    m.addConstr(12 * vitamin_b6 + 26 * fiber <= 253, "vitamin_b6_fiber_max")
    m.addConstr(18 * calcium + 12 * vitamin_b6 + 20 * magnesium <= 331, "calcium_vitamin_b6_magnesium_max")
    m.addConstr(12 * vitamin_b6 + 20 * magnesium + 26 * fiber <= 125, "vitamin_b6_magnesium_fiber_max")
    m.addConstr(kidney_support_index <= 125, "total_kidney_support_max")


    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('calcium:', calcium.x)
        print('vitamin_b6:', vitamin_b6.x)
        print('magnesium:', magnesium.x)
        print('fiber:', fiber.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)

except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
