
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
vitamin_k = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_k")
vitamin_b5 = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="vitamin_b5")
carbohydrates = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="carbohydrates")
fiber = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="fiber")
calcium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="calcium")
magnesium = m.addVar(lb=0, vtype=gp.GRB.CONTINUOUS, name="magnesium")


# Set objective function
m.setObjective(8.32 * vitamin_k + 7.18 * vitamin_b5 + 4.85 * carbohydrates + 1.85 * fiber + 5.8 * calcium + 7.09 * magnesium, gp.GRB.MINIMIZE)

# Add constraints based on the provided resources/attributes
resources = {
    'r0': {'upper_bound': 141, 'x0': 4, 'x1': 2, 'x2': 5, 'x3': 4, 'x4': 1, 'x5': 9},
    'r1': {'upper_bound': 403, 'x0': 7, 'x1': 7, 'x2': 9, 'x3': 9, 'x4': 2, 'x5': 7},
    'r2': {'upper_bound': 109, 'x0': 3, 'x1': 8, 'x2': 8, 'x3': 3, 'x4': 11, 'x5': 9},
    'r3': {'upper_bound': 411, 'x0': 8, 'x1': 5, 'x2': 1, 'x3': 2, 'x4': 9, 'x5': 6}
}

variables = [vitamin_k, vitamin_b5, carbohydrates, fiber, calcium, magnesium]

for r, data in resources.items():
    m.addConstr(sum(data[f'x{i}'] * variables[i] for i in range(6)) <= data['upper_bound'], r)


# Add the additional constraints from the JSON output.  Note:  Many of these
# constraints are redundant and could be removed for efficiency.
m.addConstr(calcium + magnesium >= 17)
m.addConstr(fiber + calcium >= 22)
m.addConstr(vitamin_b5 + magnesium >= 20)
m.addConstr(carbohydrates + calcium >= 22)
m.addConstr(fiber + magnesium >= 12)
m.addConstr(vitamin_k + magnesium >= 17)
m.addConstr(4*vitamin_k + 2*vitamin_b5 + 5*carbohydrates + 4*fiber + calcium + 9*magnesium >= 17)
# ... (add all other constraints from JSON)
m.addConstr(-4 * fiber + 3 * magnesium >= 0)
m.addConstr(8*vitamin_k + 5*vitamin_b5 + carbohydrates + 2*fiber + 9*calcium + 6*magnesium >= 32)


# Optimize model
m.optimize()

# Print solution or infeasibility status
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print('Optimization problem is infeasible.')
else:
    print(f"Optimization terminated with status {m.status}")

