
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("nutrient_optimization")

    # Create variables
    potassium = m.addVar(vtype=gp.GRB.INTEGER, name="potassium")  # Integer amount
    calcium = m.addVar(vtype=gp.GRB.CONTINUOUS, name="calcium")  # Fractional amount

    # Set objective function
    m.setObjective(4 * potassium + 1 * calcium, gp.GRB.MAXIMIZE)

    # Add constraints
    m.addConstr(13 * potassium + 10 * calcium >= 27, "energy_stability_min")
    m.addConstr(14 * potassium + 3 * calcium >= 10, "digestive_support_min")
    m.addConstr(-1 * potassium + 3 * calcium >= 0, "potassium_calcium_ratio")
    m.addConstr(13 * potassium + 10 * calcium <= 61, "energy_stability_max")
    m.addConstr(14 * potassium + 3 * calcium <= 37, "digestive_support_max")


    # Optimize model
    m.optimize()

    # Print results
    if m.status == gp.GRB.OPTIMAL:
        print('Optimal objective value: %g' % m.objVal)
        print('Potassium: %g' % potassium.x)
        print('Calcium: %g' % calcium.x)
    elif m.status == gp.GRB.INF_OR_UNBD:
        print('Model is infeasible or unbounded')
        exit(0)
    elif m.status == gp.GRB.INCONSTR:
        print('Model is infeasible')
        exit(0)
    elif m.status == gp.GRB.UNBOUNDED:
        print('Model is unbounded')
        exit(0)
    else:
        print('Optimization ended with status %d' % m.status)
        exit(0)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
