
import gurobipy as gp

# Create a new model
m = gp.Model("nutrition_optimization")

# Create variables
magnesium = m.addVar(vtype=gp.GRB.INTEGER, name="magnesium")
vitamin_k = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_k")
protein = m.addVar(vtype=gp.GRB.INTEGER, name="protein")
vitamin_c = m.addVar(vtype=gp.GRB.CONTINUOUS, name="vitamin_c")
vitamin_b1 = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_b1")
vitamin_b3 = m.addVar(vtype=gp.GRB.INTEGER, name="vitamin_b3")


# Set objective function
m.setObjective(5*magnesium + 6*vitamin_k + 9*protein + 1*vitamin_c + 1*vitamin_b1 + 4*vitamin_b3, gp.GRB.MINIMIZE)

# Resource data
resources = {
    'r0': {'upper_bound': 443, 'x0': 10.86, 'x1': 0.68, 'x2': 1.92, 'x3': 11.26, 'x4': 3.12, 'x5': 11.45},
    'r1': {'upper_bound': 346, 'x0': 1.91, 'x1': 1.13, 'x2': 2.47, 'x3': 9.52, 'x4': 10.42, 'x5': 10.9},
    'r2': {'upper_bound': 276, 'x0': 11.95, 'x1': 8.43, 'x2': 9.12, 'x3': 9.44, 'x4': 10.86, 'x5': 9.74},
    'r3': {'upper_bound': 660, 'x0': 4.28, 'x1': 5.48, 'x2': 3.32, 'x3': 11.3, 'x4': 0.44, 'x5': 10.36}
}

# Add resource constraints
for r, data in resources.items():
    m.addConstr(data['x0']*magnesium + data['x1']*vitamin_k + data['x2']*protein + data['x3']*vitamin_c + data['x4']*vitamin_b1 + data['x5']*vitamin_b3 <= data['upper_bound'], name=r)


# Add other constraints (simplified and deduplicated)
m.addConstr(11.26*vitamin_c + 3.12*vitamin_b1 >= 56)
m.addConstr(10.86*magnesium + 11.26*vitamin_c >= 71)
m.addConstr(1.92*protein + 11.26*vitamin_c >= 47)
m.addConstr(1.92*protein + 11.45*vitamin_b3 >= 61)
m.addConstr(10.86*magnesium + 11.45*vitamin_b3 >= 31)
m.addConstr(10.86*magnesium + 3.12*vitamin_b1 >= 61)
m.addConstr(1.92*protein + 3.12*vitamin_b1 >= 66)
m.addConstr(0.68*vitamin_k + 1.92*protein >= 55)
m.addConstr(3.12*vitamin_b1 + 11.45*vitamin_b3 >= 62)
m.addConstr(10.86 * magnesium + 0.68 * vitamin_k + 11.26 * vitamin_c >= 73)
m.addConstr(-4 * magnesium + 8 * vitamin_k >= 0)
m.addConstr(-6 * magnesium + 5 * vitamin_b1 >= 0)
m.addConstr(2 * magnesium - 2 * vitamin_k - 10 * protein >= 0)


# Optimize model
m.optimize()

# Print solution
if m.status == gp.GRB.OPTIMAL:
    print('Obj: %g' % m.objVal)
    for v in m.getVars():
        print('%s %g' % (v.varName, v.x))
elif m.status == gp.GRB.INFEASIBLE:
    print("Model is infeasible")
else:
    print("Optimization ended with status %d" % m.status)

