
import gurobipy as gp

try:
    # Create a new model
    m = gp.Model("resource_allocation")

    # Create variables
    light_infantry = m.addVar(vtype=gp.GRB.INTEGER, name="light_infantry")
    water_purification = m.addVar(vtype=gp.GRB.INTEGER, name="water_purification")


    # Set objective
    m.setObjective(2*light_infantry + 2*water_purification, gp.GRB.MINIMIZE)

    # Add constraints
    m.addConstr(0.37*light_infantry + 1.05*water_purification >= 22, "logistics_min")
    m.addConstr(0.37*light_infantry + 1.05*water_purification <= 35, "logistics_max")
    m.addConstr(1.57*light_infantry + 1.74*water_purification >= 22, "offensive_min")
    m.addConstr(1.57*light_infantry + 1.74*water_purification <= 38, "offensive_max")
    m.addConstr(8*light_infantry - 8*water_purification >= 0, "light_infantry_water_purification_relation")


    # Resource upper bound constraints (implicit)
    m.addConstr(light_infantry <= 65 / 0.37, "light_infantry_resource_bound_r0")
    m.addConstr(water_purification <= 65 / 1.05, "water_purification_resource_bound_r0")
    m.addConstr(light_infantry <= 50 / 1.57, "light_infantry_resource_bound_r1")
    m.addConstr(water_purification <= 50 / 1.74, "water_purification_resource_bound_r1")



    # Optimize model
    m.optimize()

    if m.status == gp.GRB.OPTIMAL:
        print('Obj: %g' % m.objVal)
        print('light_infantry:', light_infantry.x)
        print('water_purification:', water_purification.x)
    elif m.status == gp.GRB.INFEASIBLE:
        print('The model is infeasible.')
    else:
        print('Optimization ended with status %d' % m.status)


except gp.GurobiError as e:
    print('Error code ' + str(e.errno) + ': ' + str(e))

except AttributeError:
    print('Encountered an attribute error')
