from collections import Counter, defaultdict
from datetime import datetime
from itertools import combinations_with_replacement, combinations
import json
import math
from num2words import num2words
import numpy as np
import os
import random
# import re
# from sympy import symbols, Eq, Poly, Matrix, solve
# from typing import List, Dict, Union
import pickle

def get_with_lazy_default(d, key, default_fn):
    if key in d.keys():
        return d[key]
    else:
        return default_fn()
#    return d[key] if key in d else default_fn_result
class Problem:
    
    """Representation of an optimization problem as realized through terms in an E-ring.
    
    Users either provide a dictionary that populates the following attributes or otherwise we generate them using self.params.
    
    Attributes:
        num_vars (int): indicates the number of symbolic variables appearing in the problem set.
        num_resources (int): indicates the number of different resource types that will appear in the problem
        resources_dict (dict):     keys: values:: weights:A (num_vars,num_resources) matrix of scalar-valued functions (or constants) , budget_caps: a constant vector indicating maximum values for each resource.
        problem_type (dict) : determines if the problem is linear, quadratic, or general convex (of degree d), and whether it is continuous, mixed, or integer program. Provide as a dictionary dict({"degree": int(degree),"is_integer": mixed/2}), where mixed is 0,1 or 2.
        goal : str: either 'maximize' or 'minimize'
        objective_function_data: dict({"objective_function":objective_function,"term_list":term_list})), where objective function is a  string composed of summands of form "{coefficient_i} * {monomial_i}", from the term_list consisting of pairs (coefficient_i, monomial_i)
        constraints: dictionary with keys 'is_integer','non_negativity', 'lower_bound', 'upper_bound'. is_integer are equational and apply to the type for each variable x_i, non-negativity indicate a variable must be non-negative,
            and the lower and upper bound constraints are distinguished by having variables on the left-hand side, a scalar on the right hand side, and differ by '>=' for lower bounds, and '<=' for upper bounds. 
        semantic_problem_type : str: from the following list: ["office supplies", "family budget", "gardening", "network defense", "force structure", "diet0", "diet1","personnel"]
        sym_vars :a dictionary whose keys are 'x1','x2','x3',...,'xn', and whose values are a dictionary with the following key-value pair:
            ------------------------
            'full_phrase': natural-language/string substitute for the variable
            'differentiators: list of strings used
            'common_objects': (repeated) instance of commonly shared tokens/strings between
            'target': PRESENTLY DEPRECATED
            ------------------------
        problem_variables : a list of the full_phrase values for each of the sym_vars; will automatically be generated if you include the sym_vars
        nl_resources_dict :  a dictionary whose keys are 'r1','r2',...,'rm', and whose values are a dictionary with the following key-value pair:
            ------------------------
            'description': natural-language/string substitute for the variable,
            'x{i}': corresponding value determining x{i}, typically drawn from resource matrix, but conceivably a lambda expression.
            'upper_bound': upperbound constraint, maybe null.
            --=---------------------
        resources:  a list of strings describing the natural language objects extracted from resources dict using the "description" key for each resource ri
        sym_constraints :  a dict of connect the symbolic constraints described in constraints with a corresponding natural language description of struct: dict[str, list[str]], each key name corresponding to the constraint name in the constraint attribute, and each value corresponding to the symbolic representation of the constraint. 
        problem_constraints : List summarizing problem constraints, formed by substituting and reformulating the sym_constraints -- extracted as a list of the values from the self.sym_constraints attribute
        objective_function_statement : a natural language string describing the objective function.
        problem_statement: a natural language string describing the full optimization problem.
        gurobi_code: for now, intended to be a string that can execute problem -- explicitly stated as gurobi code within the internal method. Can be renamed in a future refactor.
        problem_solution: output of the gurobi code method upon execution as a dictionary with keys: 'Optimal solution', 'Value'.  Either describes the problem solution, or otherwise states problem is Infeasible
        
    Otherwise
    
    Default inputs
    -------------------
        The number of vars is determined by 6 parameters determining the shape of a mixed model of three different uniform distributions
        :nvar_r1u: int: default 4, one of 6 parameters used to determine the randomly generated number of variables for the problem
        :nvar_r2u: int: default 3, one of 6 parameters used to determine the randomly generated number of variables for the problem
        :nvar_r3u: int: default 3, one of 6 parameters used to determine the randomly generated number of variables for the problem
        :nvar_branch_1: float default 3/4, one of 6 parameters used to determine the randomly generated number of variables for the problem
        :nvar_branch_2: float default 1/3 one of 6 parameters used to determine the randomly generated number of variables for the problem
        :nvar_branch_3: float default 1/3,one of 6 parameters used to determine the randomly generated number of variables for the problem 
        :ngood_u: int: default 4, integer used to determine the hypergeometric distribution shape for number of resources, which are used to determine constraints
        :nbad_opt: Union[int,None]: optional integer to determine the hypergeometric distribution shape for number of resources, which are used to determine constraints
        :num_resource_alt: defaults as None, methods for alternative forms of resource specification for constraint generation are forthcoming.
        :resources_dict_gen: defaults as None, methods forthcoming to allow multivariate functions to define resource costs rather than a static matrix. 
        :premade_constraints: defaults as None, allows for the provision of a dictionary of premade constraints, which must have keys 'is_integer','non_negativity', 'lower_bound', 'upper_bound' and have variables indexed below the nvars value
        :deg_2_threshold: defaults .75, intended to allow for quadratic terms to appear in the constraints provided a uniform random variable exceeds the threshold. 
        :cts_threshold: default .5, intended to determine if the problem is continuous, mixed integer, or integer; type_seed, a uniform random variable below the threshold indicates the problem is mixed_integer
        :mixed_int_threshold: default .75, intended to determine if the problem is mixed integer, requires the type_seed to be above cts_threshold to be activated. If type seed is above mixed_int threshold, problem is integer valued. 
        :max_threshold: default .5, determines if the problem goal is to maximize or minimize an objective function. If uniform random variable passes threshold, the goal is to maximize, else minimize.
        :int_coefs_only_threshold: default .5, if a randomly generated uniform variable exceeds this threshold, the coefficients will only be integer values.
        :triple_sign_1_threshold: float: default 1/3, used to determine the sign value for ratios of triples appearing in constraints (see _gen_constraints()) 
        :triple_sign_2_threshold: float: default 2/3, used to determine the sign value for ratios of triples appearing in constraints (see _gen_constraints())
        :pair_sign_threshold: default 1/2, used to determine the sign value for ratios of pairs in constraints (see _gen_constraints())
        :include_exp: bool, defaults as False. If true, will enable generation of exponential-polynomial terms. 
            In this code sample, no methods are present to produce proper exponential-polynomial terms. These 
            are forthcoming with methods implementing code in SymPy instead of Gurobi in order to ensure stability
        :mheight: int, default as 0. Sets the maximum height of exponential-polynomial terms. 
    """
    
    def __init__(self, 
                nvar_r1u = 4,
                nvar_r2u = 3,
                nvar_r3u = 3, 
                nvar_branch_1 = 3/4, 
                nvar_branch_2 = 1/3, 
                nvar_branch_3 = 1/3, 
                ngood_u = 4, 
                nbad_opt = None, 
                num_resource_alt = None, 
                resources_dict_gen = None,
                premade_constraints = None, 
                deg_2_threshold = .75,
                cts_threshold = .5,
                mixed_int_threshold = .75,
                max_threshold = .5,
                int_coefs_only_threshold = .5,
                triple_sign_1_threshold = 1/3,
                triple_sign_2_threshold = 2/3,
                pair_sign_threshold = 1/2,
                include_exp = False,
                mheight = 0,
                **problem_dict):
        """Randomly generates an instance of an optimization problem"""
        self.params = dict({
            "nvar_r1u":nvar_r1u,
            "nvar_r2u":nvar_r2u,
            "nvar_r3u":nvar_r3u, 
            "nvar_branch_1": nvar_branch_1, 
            "nvar_branch_2":nvar_branch_2, 
            "nvar_branch_3":nvar_branch_3, 
            "ngood_u":ngood_u, 
            "nbad_opt" : nbad_opt, 
            "num_resource_alt" : num_resource_alt, 
            "resources_dict_gen" : resources_dict_gen, 
            "deg_2_threshold" : deg_2_threshold,
            "cts_threshold" : cts_threshold,
            "mixed_int_threshold" :mixed_int_threshold,
            "max_threshold" : max_threshold,
            "int_coefs_only_threshold": int_coefs_only_threshold,
            "triple_sign_1_threshold": triple_sign_1_threshold,
            "triple_sign_2_threshold": triple_sign_2_threshold,
            "pair_sign_threshold" : pair_sign_threshold,
            "include_exp" : include_exp,
            "mheight" : mheight,
            "premade_constraints" : premade_constraints,
        })
        self.num_vars = get_with_lazy_default(problem_dict, "num_vars", self._gen_num_vars)#problem_dict.get('num_vars',self._gen_num_vars())
        self.num_resources = get_with_lazy_default(problem_dict,'num_resources',self._gen_num_resources)
        self.resources_dict = get_with_lazy_default(problem_dict,'resources_dict',self._gen_resources_dict)
        self.problem_type = get_with_lazy_default(problem_dict,'problem_type',self._gen_problem_type)
        self.goal = get_with_lazy_default(problem_dict,'goal',self._gen_goal)
        self.objective_function_data = get_with_lazy_default(problem_dict,'objective_function_data',self._gen_function)
        self.constraints = get_with_lazy_default(problem_dict,'constraints',self._gen_constraints)
        # ATTRIBUTE AND METHOD FOR GENERATING SEMANTIC CONTEXT OF THE REMAINING ATTRIBUTES
        self.semantic_problem_type = get_with_lazy_default(problem_dict,'semantic_problem_type',self._select_semantic_problem)
        self.sym_vars = get_with_lazy_default(problem_dict,'sym_vars',self._gen_sym_vars)
        self.problem_variables = get_with_lazy_default(problem_dict,'problem_variables',self._get_problem_variables)
        self.nl_resources_dict = get_with_lazy_default(problem_dict,'nl_resources_dict',self._gen_nl_resources_dict) # Dictionary of natural language expressions describing the resources consumed in constraints (used to bound the constraint types)
        self.resources = get_with_lazy_default(problem_dict,'resources',self._gen_resources_list)
        self.sym_constraints = get_with_lazy_default(problem_dict,'sym_constraints', self._gen_sym_constraints) # Dictionary of symbolic variables connecting variables to natural language descriptions appearing in the constraints
        self.problem_constraints = get_with_lazy_default(problem_dict,'problem_constraints',self._gen_problem_constraints) 
        self.objective_function_statement = get_with_lazy_default(problem_dict,'objective_function_statement',self._gen_objective_function_statement) # String natural language description of objective function, substituting in variables with corresponding string in sym_vars dictionary
        self.problem_statement = get_with_lazy_default(problem_dict,"problem_statement",self._gen_problem_statement) # String describing problem statement
        self.gurobi_code = problem_dict.get('gurobi_code',self._gen_gurobi_code())
        self.problem_solution = problem_dict.get('problem_solution',self._run_gurobi_code())

    def _gen_num_vars(self):
        """Randomly generates a non-negative integer greater than 1."""
        r1 = random.randint(2,self.params['nvar_r1u'])
        r2 = random.randint(0,self.params['nvar_r2u'])
        r3 = random.randint(0,self.params['nvar_r3u'])
        s = random.random()
        t = random.random()
        r = r1
        if s > self.params['nvar_branch_1']:
            r += r2
        if t >= s + self.params['nvar_branch_2']:
            r += r3 
        if s >= t + self.params['nvar_branch_3']:
            r += max(r2 - r3, r3 - r2, 1)
        return r
    
    def _gen_num_resources(self):
        """Randomly generates a non-negative integer greater than 0"""
        ngood = random.randint(1, self.params['ngood_u'])
        nbad = self.params['nbad_opt'] if not isinstance(self.params['nbad_opt'], type(None)) else random.randint(2*ngood, 4*ngood)
        s = random.random()
        if self.params['num_resource_alt']:
            return self.params['num_resource_alt']
        else:
            if ngood >= s*(nbad-1):
                return 1 + np.random.hypergeometric(ngood,nbad,random.randint(nbad,ngood + nbad))
            else: 
                return 1 + np.random.hypergeometric(ngood,nbad,random.randint(ngood,2*ngood))
    
    def _gen_resources_dict(self,resources_split_parameter):
        """Randomly generates a (num_vars,num_resources) matrix of integral or half-integral coefficients """
        if self.params['resources_dict_gen']:
            return self.resources_dict_gen
        else:
            a=random.randint(2,10)
            b=random.randint(2,10)
            t = 2+np.random.hypergeometric(a,b,random.randint(a,a+b))
            if random.random()>resources_split_parameter:       
                return dict({"weights":np.random.randint(1,3*t,(self.num_vars,self.num_resources)),
                    "budget_caps":np.random.randint(random.randint(2*t*self.num_vars,5*t*self.num_vars),random.randint(6*t*self.num_vars,20*t*self.num_vars),self.num_resources)})
            else:
                return dict({"weights":np.round(np.random.rand(self.num_vars,self.num_resources)*random.randint(1,3*t),2),
                    "budget_caps":np.random.randint(random.randint(2*t*self.num_vars,5*t*self.num_vars),random.randint(6*t*self.num_vars,20*t*self.num_vars),self.num_resources)})
    
    def _gen_problem_type(self):
        """Randomly determines if the problem is linear, quadratic, or general convex (of degree d), and whether it is continuous, mixed, or integer program."""
        degree_seed = random.random()
        # if degree_seed> .95:
        #     ngood = random.randint(1,10)
        #     nbad = random.randint(1,10)
        #     nsample = random.randint(max(ngood-1,nbad-1,1),ngood+nbad-1)
        #     degree = 2 + np.random.hypergeometric(ngood,nbad,nsample)
        # elif
        if degree_seed >self.params['deg_2_threshold']:
            degree = 2
        else:
            degree = 1
        type_seed = random.random()
        if degree>1:
            if type_seed>1/degree:
                mixed = random.randint(0,2)
            else:
                mixed = random.randint(0,1)
        else:
            if type_seed < self.params['cts_threshold']:
                mixed = 0
            elif type_seed <self.params['mixed_int_threshold']:
                mixed = 1
            else:
                mixed = 2
        problem_type = dict({"degree": int(degree),
                            "is_integer": mixed/2,            
        })
        return problem_type
    
    def _gen_goal(self):
        if random.random()>self.params['max_threshold']:
            return "maximize"
        else:
            return "minimize"
        
    def _gen_function(self):
        """Generate the objective function"""
        var_list = ["x"+str(n) for n in range(self.num_vars)]
        if self.params["include_exp"]:
            # We have removed functionality for now building arbitrary towers of exp-poly functions for public use
            print("WARNING: Exp-poly terms are disabled at this time for public use.")
            pass
        else:
            upper_bound = 0
            degree_grading= []
            D = self.problem_type['degree']
            for d in range(D,0,-1):
                degree_grading.append(math.comb(d+self.num_vars-1,self.num_vars-1))
                upper_bound += math.comb(d+self.num_vars-1,self.num_vars-1)
            lower_bound = self.num_vars
            num_of_coefficients = random.randint(lower_bound,upper_bound)
            p_int_only = random.random()
            if p_int_only>self.params['int_coefs_only_threshold']:
                coefficients = np.random.randint(low=1,high=10,size=num_of_coefficients)
            else:
                coefficients = np.random.randint(low=1,high=10,size=num_of_coefficients)+np.random.random(size=num_of_coefficients)
            for i in range(len(coefficients)):
                if coefficients[i] <0.01:
                    coefficients[i] = 0.01
            # Generate binary vectors for descending homogeneous degree terms so that at least half of the top terms are in the 
            monomial_selector = generate_boolean_vectors(degree_grading,num_of_coefficients)
            monomial_list = []
            # full degree and the total length of trues is the number of coefficients
            # Produce the monomial term list from the implied multiindex using the binary vector terms
            for i, selector in enumerate(monomial_selector):
                monos = generate_monomials(var_list,int(self.problem_type['degree']-i))
                monomial_list+= [mono for i,mono in enumerate(monos) if selector[i]]        # wherefore selector?
            # Produce the term list in descending degree by xi_0 *x_i1 *...*x_id (ie. short cut the multiindex selection from the binary vectors list)
            # by zipping the coefficients with the monomial term list
            term_list = list(zip(coefficients,monomial_list))  # iterators are exhausted after iteration, must use list to save later
            # Concatenate the term list into a string
            objective_function = ""
            for (coef,mono) in term_list:
                objective_function += f"{coef:.2f} * {mono} + "
            objective_function = objective_function[:-3]
            return dict({"objective_function": objective_function, "term_list": term_list})
    
    def _gen_constraints(self):
        """Returns a dictionary of dictionaries of constraints with the following output key-values. All inner constraint keys are numbered Ci, where i corresponds to
        the specific constraint number, optionally followed by underscores, _<resource number> or _<type>, indicating the resource number, and type of constraint (non-negativity, upper-bound or lower bound, type identity, and upper/lower bound with tradeoffs)
        
        For ease of use later, we output all the constraints as a dictionary of four dictionaries of dictionaries across our four principle constraint types, each innermost dictionary has the following key-value pair
        
        Inner key-value
        -------------------------
        :full: a string describing the full constraint 
        :tuple_list: (optional) a list of the coefficient and monomial appearing in the expression
        :relation: one of the following: '>=','<=','=='
        :bound:  a number or string indicating type int or type float 
        
        Output key-value
        ------------------
        :is_integer: Constraints establishing if a variable is integral
        :non_negativity: Necessary constraints
        :lower_bound: Additional lower bound constraints describing trade-offs between several variables
        :upper_bound: Necessary upper bound constraints
        
        Lower and upper bound constraints are either determined solely from the resource coefficient matrix if the bounds are
        non-zero, or are purely integral relations if the bound is chosen to be 0, indicating an exchange relationship between
        the variables.
        """
        # TO DO: REWRITE FOR EXPONENTIAL CONSTRAINTS TO RANDOMLY APPEAR WITHIN METHOD IF NOT INTEGER PROBLEM
        if self.params['premade_constraints']:
            return self.params['premade_constraints']
        else:
            vars = ["x"+str(n) for n in range(self.num_vars)]
            counter = 0
            non_negativity = dict()
            lower_bound = dict()
            upper_bound = dict()
            int_types = dict()

            for i in range(0,len(vars)):
                non_negativity.__setitem__('C'+str(i),dict({"description":vars[i]+" >= 0",
                                                            'tuple_list':[(1,vars[i])],
                                                            'relation':">=",
                                                            "bound":0}))
                counter+=1
            # Generate random number of additional bounds
            # Select additional random number of lower bounds composed of pairs
            if self.problem_type['is_integer'] == 1:
                for var in vars:
                    int_types.__setitem__('C_t_'+str(var),dict({'full':f"{var} == int",'relation':'==','bound':'int'}))
                    counter += 1
            elif self.problem_type['is_integer'] == 1/2:
                sample_num = random.randint(1,len(vars)-1)
                int_sample = random.sample(vars,sample_num)
                for sample in vars:
                    if sample in int_sample:
                        int_types.__setitem__('C_t_'+str(sample),dict({'full':f"{sample} == int",'relation':'==','bound':'int'}))
                    else:
                        int_types.__setitem__('C_t_'+str(sample),dict({'full':f"{sample} == float",'relation':'==','bound':'float'}))
            else:
                for sample in vars:
                    int_types.__setitem__('C_t_'+str(sample),dict({'full':f"{sample} == float",'relation':'==','bound':'float'}))
            total_binder = random.randint(2*len(vars),100*len(vars))            
            for j in range(self.num_resources):
                if len(vars)>2:
                    if self.goal == 'minimize':
                        number_of_lpairs = random.randint(len(vars),math.comb(len(vars),2))
                        number_of_ltriples = random.randint(0,math.comb(len(vars),3))
                        number_of_upairs = random.randint(0,math.comb(len(vars),2))
                        number_of_utriples = random.randint(0,math.comb(len(vars),3))
                    else:
                        number_of_upairs = random.randint(len(vars),math.comb(len(vars),2))
                        number_of_utriples = random.randint(0,math.comb(len(vars),3))
                        number_of_lpairs = random.randint(0,math.comb(len(vars),2))
                        number_of_ltriples = random.randint(0,math.comb(len(vars),3))
                    number_of_lpairs = np.random.hypergeometric(number_of_lpairs, math.comb(len(vars),2),(3*number_of_lpairs)//2) if number_of_lpairs !=0 else 0
                    number_of_ltriples = np.random.hypergeometric(number_of_ltriples, math.comb(len(vars),3),(3*number_of_ltriples)//2) if number_of_ltriples !=0 else 0
                    number_of_upairs = np.random.hypergeometric(number_of_upairs, math.comb(len(vars),2),(3*number_of_upairs)//2) if number_of_upairs !=0 else 0
                    number_of_utriples = np.random.hypergeometric(number_of_utriples, math.comb(len(vars),3),(3*number_of_utriples)//2) if number_of_utriples !=0 else 0
                else:
                    number_of_lpairs = 1
                    number_of_upairs = 1
                    number_of_ltriples = 0
                    number_of_utriples = 0
                    
                # Sample for lower/upper pairs and triples
                lpairs = random.sample([combo for combo in combinations(vars,2)], number_of_lpairs)
                ltriples = random.sample([combo for combo in combinations(vars,3)], number_of_ltriples) if number_of_ltriples !=0 else []
                upairs = random.sample([combo for combo in combinations(vars,2)], number_of_upairs)
                utriples = random.sample([combo for combo in combinations(vars,3)], number_of_utriples) if number_of_utriples !=0 else[]
                # print(f"DEBUG STATEMENTS:\n The lpairs are {lpairs}\n The ltriples are {ltriples}\n The upairs are {upairs}. \n The utriples are {utriples}.")
                # Set the additional lower bounds
                for i in range(0,len(lpairs)):
                    constraint_dict = dict()
                    try:
                        coeff1 = self.resources_dict['weights'][int(lpairs[i][0][1:])][j]
                        coeff2 = self.resources_dict['weights'][int(lpairs[i][1][1:])][j]
                    except Exception as E:
                        print(f"Had error {E}")
                        print(self.resources_dict['weights'])
                        print(f'{lpairs[i]}')
                    bound = random.randint(self.resources_dict['budget_caps'][j]//(3*self.num_vars),self.resources_dict['budget_caps'][j]//self.num_vars)
                    # coeff1 = random.randint(1,total_binder//len(vars))
                    # coeff2 = random.randint(-2*coeff1,2*coeff1)
                    if self.problem_type['degree'] == 1:
                        constraint_dict.__setitem__('description',f"{coeff1} * {lpairs[i][0]} + {coeff2} * {lpairs[i][1]} >= {bound}")
                        constraint_dict.__setitem__('tuple_list',[(coeff1,lpairs[i][0]),(coeff2,lpairs[i][1])])
                        constraint_dict.__setitem__('relation',">=")
                        constraint_dict.__setitem__('bound',bound)
                        lower_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_lb",constraint_dict) 
                    else:
                        if random.random() > 2/((self.problem_type['degree'])+1):
                            constraint_dict.__setitem__('description',f"{coeff1} * {lpairs[i][0]} ** 2 + {coeff2} * {lpairs[i][1]} ** 2 >= {bound}")
                            constraint_dict.__setitem__('tuple_list',[(coeff1,f"{lpairs[i][0]} ** 2"),(coeff2,f"{lpairs[i][1]} ** 2")])
                            constraint_dict.__setitem__('relation',">=")
                            constraint_dict.__setitem__('bound',bound)
                            lower_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_lb", constraint_dict)
                        else:
                            constraint_dict.__setitem__('description',f"{coeff1} * {lpairs[i][0]} + {coeff2} * {lpairs[i][1]} >= {bound}")
                            constraint_dict.__setitem__('tuple_list',[(coeff1,f"{lpairs[i][0]}"),(coeff2, f"{lpairs[i][1]}")])
                            constraint_dict.__setitem__('relation',">=")
                            constraint_dict.__setitem__('bound',bound)
                            lower_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_lb", constraint_dict)
                    
                    counter += 1
                # Set the additional lower triples
                for i in range(0,len(ltriples)):
                    try:
                        coeff1 = self.resources_dict['weights'][int(ltriples[i][0][1:])][j]
                        coeff2 = self.resources_dict['weights'][int(ltriples[i][1][1:])][j]
                        coeff3 = self.resources_dict['weights'][int(ltriples[i][2][1:])][j]
                    except Exception as E:
                        print(f"Has error {E}")
                        print(f"ltriples are {ltriples} with positions i={i} and j={j}")
                    bound = random.randint(self.resources_dict['budget_caps'][j]//(2*self.num_vars),self.resources_dict['budget_caps'][j]//self.num_vars)
                    for i in range(0,number_of_ltriples):
                        constraint_dict = dict()
                        if self.problem_type['degree'] == 1:
                            constraint_dict.__setitem__('description',f"{coeff1} * {ltriples[i][0]} + {coeff2} * {ltriples[i][1]}  + {coeff3} * {ltriples[i][2]} >= {bound}")
                            constraint_dict.__setitem__('tuple_list',[(coeff1,ltriples[i][0]),(coeff2,ltriples[i][1]),(coeff3,ltriples[i][2])])
                            constraint_dict.__setitem__('relation',">=")
                            constraint_dict.__setitem__('bound',bound)
                            lower_bound.__setitem__("C"+str(counter)+"_"+str(j), constraint_dict)
                        else:
                            if random.random() > 2/((self.problem_type['degree'])+1):
                                constraint_dict.__setitem__('description',f"{coeff1*coeff1} * {ltriples[i][0]} ** 2 + {coeff2*coeff2} * {ltriples[i][1]} ** 2 + {coeff3*coeff3} * {ltriples[i][2]} ** 2 >= {bound}")
                                constraint_dict.__setitem__('tuple_list',[(coeff1*coeff1,f"{ltriples[i][0]} ** 2"),(coeff2*coeff2,f"{ltriples[i][1]} ** 2"),(coeff3*coeff3,f"{ltriples[i][2]} ** 2")])
                                constraint_dict.__setitem__('relation',">=")
                                constraint_dict.__setitem__('bound',bound)
                                lower_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_lb",constraint_dict )
                            else:
                                constraint_dict.__setitem__('description',f"{coeff1} * {ltriples[i][0]} + {coeff2} * {ltriples[i][1]} + {coeff3} * {ltriples[i][2]} >= {bound}")
                                constraint_dict.__setitem__('tuple_list',[(coeff1,f"{ltriples[i][0]}"),(coeff2,f"{ltriples[i][1]}"),(coeff3,ltriples[i][2])])
                                constraint_dict.__setitem__('relation',">=")
                                constraint_dict.__setitem__('bound',bound)
                                lower_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_lb",constraint_dict)
                        counter += 1
                        
                # Set the upper pairs
                for i in range(0,len(upairs)):
                    coeff1 = self.resources_dict['weights'][int(upairs[i][0][1:])][j]
                    coeff2 = self.resources_dict['weights'][int(upairs[i][1][1:])][j]
                    constraint_dict = dict()
                    bound = random.randint(self.resources_dict['budget_caps'][j]//self.num_vars,self.resources_dict['budget_caps'][j])
                    
                    if self.problem_type['degree'] == 1:
                        constraint_dict.__setitem__('description', f"{coeff1} * {upairs[i][0]} + {coeff2} * {upairs[i][1]} <= {bound}" )
                        constraint_dict.__setitem__('bound',bound)
                        constraint_dict.__setitem__('relation', '<=')
                        constraint_dict.__setitem__('tuple_list', [(coeff1,upairs[i][0]),(coeff2,upairs[i][1])])                    
                        upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_ub",constraint_dict)
                    else:
                        if random.random() > 2/((self.problem_type['degree'])+1):
                            constraint_dict.__setitem__('description', f"{coeff1 *coeff1} * {upairs[i][0]} ** 2 + {coeff2*coeff2} * {upairs[i][1]} ** 2 <= {bound}")
                            constraint_dict.__setitem__('bound',bound)
                            constraint_dict.__setitem__('relation', '<=')
                            constraint_dict.__setitem__('tuple_list', [(coeff1 *coeff1,f"{upairs[i][0]} ** 2"),(coeff2*coeff2,f"{upairs[i][1]} ** 2")])                    
                            upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_ub",constraint_dict)
                        else:
                            constraint_dict.__setitem__('description', f"{coeff1} * {upairs[i][0]} + {coeff2} * {upairs[i][1]} <= {bound}" )
                            constraint_dict.__setitem__('bound',bound)
                            constraint_dict.__setitem__('relation', '<=')
                            constraint_dict.__setitem__('tuple_list', [(coeff1,upairs[i][0]),(coeff2,upairs[i][1])])                    
                            upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_ub",constraint_dict)
                    counter += 1
                # Set the upper triples
                for i in range(0,len(utriples)):
                    constraint_dict = dict()
                    coeff1 = self.resources_dict['weights'][int(utriples[i][0][1:])][j]
                    coeff2 = self.resources_dict['weights'][int(utriples[i][1][1:])][j]
                    coeff3 = self.resources_dict['weights'][int(utriples[i][2][1:])][j]
                    bound = random.randint(self.resources_dict['budget_caps'][j]//self.num_vars,self.resources_dict['budget_caps'][j])
                    if self.problem_type['degree'] == 1:
                        constraint_dict.__setitem__('description',f"{coeff1} * {utriples[i][0]} + {coeff2} * {utriples[i][1]}  + {coeff3} * {utriples[i][2]} <= {bound}")
                        constraint_dict.__setitem__('bound',bound)
                        constraint_dict.__setitem__('relation', '<=')
                        constraint_dict.__setitem__('tuple_list', [(coeff1,utriples[i][0]),(coeff2,utriples[i][1]),(coeff3,utriples[i][2])])                    
                        upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_ub",constraint_dict)
                    else:
                        if random.random() > 2/((self.problem_type['degree'])+1):
                            constraint_dict.__setitem__('description',f"{coeff1*coeff1} * {utriples[i][0]} ** 2 + {coeff2*coeff2} * {utriples[i][1]} ** 2 + {coeff3*coeff3} * {utriples[i][2]} <= {bound}")
                            constraint_dict.__setitem__('bound',bound)
                            constraint_dict.__setitem__('relation', '<=')
                            constraint_dict.__setitem__('tuple_list', [(coeff1*coeff1,f"{utriples[i][0]} ** 2"),(coeff2*coeff2,f"{utriples[i][1]} ** 2"),(coeff3*coeff3,f"{utriples[i][2]} ** 2")])                    
                            upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_ub",constraint_dict)

                        else:
                            constraint_dict.__setitem__('description',f"{coeff1} * {utriples[i][0]} + {coeff2} * {utriples[i][1]}  + {coeff3} * {utriples[i][2]} <= {bound}")
                            constraint_dict.__setitem__('bound',bound)
                            constraint_dict.__setitem__('relation', '<=')
                            constraint_dict.__setitem__('tuple_list', [(coeff1,utriples[i][0]),(coeff2,utriples[i][1]),(coeff3,utriples[i][2])])                    
                            upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_ub",constraint_dict)
                    counter += 1
                final_sum = ""
                constraint_dict = dict()
                tuple_list = []
                for i,var in enumerate(vars):
                    final_sum += f"{self.resources_dict['weights'][i][j]:.2f} * {var} + "
                    tuple_list.append((f"{self.resources_dict['weights'][i][j]:.2f}",var))
                final_sum = final_sum[:-3]
                constraint_dict.__setitem__('tuple_list',tuple_list)
                if self.goal == "minimize":
                    constraint_dict.__setitem__('bound', f"{self.resources_dict['budget_caps'][j]/2:.2f}")
                    constraint_dict.__setitem__('description',f"{final_sum} >= {self.resources_dict['budget_caps'][j]/2:.2f}")
                    constraint_dict.__setitem__('relation', ">=")
                    lower_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_budget",constraint_dict)
                else:
                    constraint_dict.__setitem__('bound', f"{self.resources_dict['budget_caps'][j]:.2f}")
                    constraint_dict.__setitem__('description',f"{final_sum} <= {self.resources_dict['budget_caps'][j]/2:.2f}")
                    constraint_dict.__setitem__('relation', "<=")
                    upper_bound.__setitem__("C"+str(counter)+"_"+str(j)+"_budget",constraint_dict)
            
            # Now add interchange constraints
            if len(vars)>2:
                if self.goal == 'minimize':
                    number_of_lpairs = random.randint(len(vars),max(math.comb(len(vars),2)//len(vars),len(vars)))
                    number_of_ltriples = random.randint(0,math.comb(len(vars),3)//len(vars))
                    number_of_upairs = random.randint(0,math.comb(len(vars),2)//len(vars))
                    number_of_utriples = random.randint(0,math.comb(len(vars),3)//len(vars))
                else:
                    number_of_upairs = random.randint(len(vars),max(math.comb(len(vars),2)//len(vars),len(vars)))
                    number_of_utriples = random.randint(0,math.comb(len(vars),3)//len(vars))
                    number_of_lpairs = random.randint(0,math.comb(len(vars),2)//len(vars))
                    number_of_ltriples = random.randint(0,math.comb(len(vars),3)//len(vars))
                number_of_lpairs = np.random.hypergeometric(number_of_lpairs, math.comb(len(vars),2),number_of_lpairs) if number_of_lpairs !=0 else 0
                number_of_ltriples = np.random.hypergeometric(number_of_ltriples, math.comb(len(vars),3),number_of_ltriples) if number_of_ltriples != 0  else 0
                number_of_upairs = np.random.hypergeometric(number_of_upairs, math.comb(len(vars),2),number_of_upairs) if number_of_upairs != 0 else 0
                number_of_utriples = np.random.hypergeometric(number_of_utriples, math.comb(len(vars),3),number_of_utriples) if number_of_utriples !=0 else 0
            else:
                number_of_lpairs = 1
                number_of_upairs = 1
                number_of_ltriples = 0            
                number_of_utriples = 0
                    
                # Sample for lower/upper pairs and triples
            lpairs = random.sample([combo for combo in combinations(vars,2)], number_of_lpairs)
            ltriples = random.sample([combo for combo in combinations(vars,3)], number_of_ltriples)
            upairs = random.sample([combo for combo in combinations(vars,2)], number_of_upairs)
            utriples = random.sample([combo for combo in combinations(vars,3)], number_of_utriples)
            # print(f"DEBUG STATEMENTS:\n The lpairs are {lpairs}\n The ltriples are {ltriples}\n The upairs are {upairs}. \n The utriples are {utriples}.")
            # Set the additional lower bounds
            for i in range(0,len(lpairs)):
                constraint_dict = dict()
                coeff1 = random.randint(1,10)
                coeff2 = random.randint(1,10)
                if random.random()>.5:
                    coeff2 = -coeff2
                else:
                    coeff1 = -coeff1
                bound = 0
                # coeff1 = random.randint(1,total_binder//len(vars))
                # coeff2 = random.randint(-2*coeff1,2*coeff1)
                if self.problem_type['degree'] == 1:
                    constraint_dict.__setitem__('description', f"{coeff1} * {lpairs[i][0]} + {coeff2} * {lpairs[i][1]} >= {bound}")
                    constraint_dict.__setitem__('relation',">=")
                    constraint_dict.__setitem__('bound',bound)
                    constraint_dict.__setitem__('tuple_list',[(coeff1,lpairs[i][0]),(coeff2,lpairs[i][1])])
                    lower_bound.__setitem__("C"+str(counter)+"_proportion",constraint_dict)
                else:
                    if random.random() > 2/((self.problem_type['degree'])+1):
                        constraint_dict.__setitem__('description',f"{coeff1} * {lpairs[i][0]} ** 2 + {coeff2} * {lpairs[i][1]} ** 2 >= {bound}")
                        constraint_dict.__setitem__('relation',">=")
                        constraint_dict.__setitem__('bound',bound)
                        constraint_dict.__setitem__('tuple_list',[(coeff1,f"{lpairs[i][0]} ** 2"),(coeff2,f"{lpairs[i][1]} ** 2")])
                        lower_bound.__setitem__("C"+str(counter)+"_proportion",constraint_dict)
                    else:
                        constraint_dict.__setitem__('description', f"{coeff1} * {lpairs[i][0]} + {coeff2} * {lpairs[i][1]} >= {bound}")
                        constraint_dict.__setitem__('relation',">=")
                        constraint_dict.__setitem__('bound',bound)
                        constraint_dict.__setitem__('tuple_list',[(coeff1,lpairs[i][0]),(coeff2,lpairs[i][1])])
                        lower_bound.__setitem__("C"+str(counter)+"_proportion",constraint_dict)            
                counter += 1
            # Set the additional lower triples
            for i in range(0,len(ltriples)):
                coeff1 = random.randint(1,10)
                coeff2 = random.randint(1,10)
                coeff3 = random.randint(1,10)
                bound = 0
                s = random.random()
                t = random.random()
                constraint_dict = dict()
                if s<self.params['triple_sign_1_threshold']:
                    if t>self.params['pair_sign_threshold']:
                        coeff2 = -coeff2
                    else:
                        coeff2 = -coeff2
                        coeff3 = -coeff3
                elif s<self.params['triple_sign_2_threshold']:
                    if t>self.params['pair_sign_threshold']:
                        coeff3 = -coeff3
                    else:
                        coeff1 = -coeff1
                else:
                    if t>self.params['pair_sign_threshold']:
                        coeff1 = -coeff1
                    else:
                        coeff1 = -coeff1
                        coeff2 = -coeff2
                if self.problem_type['degree'] == 1:
                    constraint_dict.__setitem__('description',f"{coeff1} * {ltriples[i][0]} + {coeff2} * {ltriples[i][1]}  + {coeff3} * {ltriples[i][2]} >= {bound}")
                    constraint_dict.__setitem__('relation',">=")
                    constraint_dict.__setitem__('bound',bound)
                    constraint_dict.__setitem__('tuple_list',[(coeff1,ltriples[i][0]),(coeff2,ltriples[i][1]),(coeff3,ltriples[i][2])])
                    lower_bound.__setitem__("C"+str(counter)+"_proportion",constraint_dict)            

                else:
                    if random.random() > 2/((self.problem_type['degree'])+1):
                        constraint_dict.__setitem__('description',f"{coeff1} * {ltriples[i][0]} ** 2 + {coeff2} * {ltriples[i][1]} ** 2 + {coeff3} * {ltriples[i][2]} ** 2 >= {bound}")
                        constraint_dict.__setitem__('relation',">=")
                        constraint_dict.__setitem__('bound',bound)
                        constraint_dict.__setitem__('tuple_list',[(coeff1,f"{ltriples[i][0]} ** 2"),(coeff2,f"{ltriples[i][1]} ** 2"),(coeff3,f"{ltriples[i][2]} ** 2")])
                        lower_bound.__setitem__("C"+str(counter)+"_proportion",constraint_dict)            
                    else:
                        constraint_dict.__setitem__('description',f"{coeff1} * {ltriples[i][0]} + {coeff2} * {ltriples[i][1]}  + {coeff3} * {ltriples[i][2]} >= {bound}")
                        constraint_dict.__setitem__('relation',">=")
                        constraint_dict.__setitem__('bound',bound)
                        constraint_dict.__setitem__('tuple_list',[(coeff1,ltriples[i][0]),(coeff2,ltriples[i][1]),(coeff3,ltriples[i][2])])
                        lower_bound.__setitem__("C"+str(counter)+"_proportion",constraint_dict)            
                counter += 1
            return dict({'non_negativity':non_negativity,'lower_bound':lower_bound,'upper_bound':upper_bound,'int_types':int_types,'total_binder':total_binder})

            #return dict({'non_negativivity':non_negativity,'lower_bound':lower_bound,'upper_bound':upper_bound,'int_types':int_types,'total_binder':total_binder})
    
    def _select_semantic_problem(self):
        """
        Selects a semantic problem type for determining.
        Supports up to 12 vars.
        """

        if self.problem_type["is_integer"] == 1:  # Integer
            semantic_problem_options = ["office supplies", "family budget", "gardening", "network defense", "force structure", "diet0",
                                        "personnel"
                                        ]

        elif self.problem_type["is_integer"] == 0.5:  # Mixed integer
            semantic_problem_options = ['diet0', 'diet1',
                                        'personnel',
                                        ]

        elif self.problem_type["is_integer"] == 0:  # continuous/linear
            semantic_problem_options = ["diet0", "diet1",
                                        "personnel",
                                        ]

        return random.choice(semantic_problem_options)

    def _gen_sym_vars(self):
        """Method for producing a dictionary whose keys are 'x1','x2','x3',...,'xn', and whose values are a dictionary with the following key-value pair:
        ------------------------
        'full_phrase': natural-language/string substitute for the variable
        'differentiators: list of strings used
        'common_objects': (repeated) instance of commonly shared tokens/strings between
        'target':
        """

        if self.semantic_problem_type == "office supplies":
            possible_semantic_vars = ["color printers", "monochrome printers", "3D printers", "blue pens", "red pens",
                                    "black pens", "staplers", "manila envelopes", "planners", "usb flash drives",
                                    "rubber bands", "headsets", "paper clips", "postage stamps",
                                    "lanyards", "office chairs", "cafeteria chairs", "cans of coffee",
                                    "wooden pencils", "mechanical pencils", "packs of paper", "scissors",
                                    "hole punches", "smoke detectors", "yellow highlighters", "blue highlighters",
                                    "red highlighters"
                                    ]

        elif self.semantic_problem_type == "family budget":
            possible_semantic_vars = ["packs of napkins", "cartons of milk", "cookies", "bottles of ibuprofen",
                                    "rubber gloves", "dish soap bottles", "packs of paper plates", "lightbulbs",
                                    "toilet paper rolls", "paper towel rolls", "diapers", "candles"
                                    ]

        elif self.semantic_problem_type == "gardening":
            possible_semantic_vars = ["lilies", "petunias", "geraniums", "daisies",
                                    "chrysanthemums", "hydrangeas", "begonias", "verbenas",
                                    "pansies", "vincas", "coleus", "roses",
                                    "carnations", "sunflowers", "tulips", "peonies",
                                    "boxwoods", "ferns", "decorative cabbages", "potato vines",
                                    "agave", "aloe vera", "chives", "carrots",
                                    "tomato vines", "basil plants", "cucumber vines", "chili plants",
                                    "bean stalks", "squash plants", "zucchini vines", "strawberry bushes",
                                    "orange trees", "apple trees", "cherry trees"
                                    ]

        elif self.semantic_problem_type == "network defense":
            possible_semantic_vars = ["network administrators", "intrusion analysts", "SOC operators", "security engineers",
                                    "system administrators", "pen testers", "deployed decoys", "honeypots",
                                    "security onions", "patches per day", "Mbps bandwidth allocated to monitoring", "automatic alerts"
                                    ]

        elif self.semantic_problem_type == "force structure":
            possible_semantic_vars = ["armored companies", "mechanized infantry companies", "artillery batteries", "engineer platoons",
                                    "logistics companies", "medical platoons", "signal platoons", "reconnaissance troops",
                                    "air defense batteries", "light infantry companies", "water purification units", "military intelligence companies",
                                    "transportation companies", "airborne infantry companies", "pathfinder teams", "CBRN platoons"
                                    ]

        elif self.semantic_problem_type == "personnel":
            possible_semantic_vars = ["hours worked by John", "hours worked by Paul", "hours worked by George", "hours worked by Ringo",
                                    "hours worked by Hank", "hours worked by Bobby", "hours worked by Peggy", "hours worked by Dale",
                                    "hours worked by Bill", "hours worked by Laura", "hours worked by Jean", "hours worked by Mary"
                                    ]

        elif self.semantic_problem_type == "diet0":
            possible_semantic_vars = ["eggs", "apple pies", "cherry pies", "blueberry pies", "chicken breasts",
                                    "chicken thighs", "chicken drumsticks", "pickles", "kale salads", "fruit salads",
                                    "apples", "lemons", "hamburgers", "cheeseburgers", "rotisserie chickens",
                                    "steaks", "ravioli", "milkshakes", "granola bars", "protein bars",
                                    "oreos", "oranges", "bananas", "strawberries", "black beans",
                                    "kiwis", "cantaloupes", "green beans", "bowls of instant ramen",
                                    "bowls of cereal", "strips of bacon", "bowls of pasta",
                                    "peanutbutter sandwiches", "bagged salads", "cornichons", "ham sandwiches",
                                    "tomatoes", "hot dogs", "potatoes", "sashimi", "knishes", "slices of pizza",
                                    "corn cobs"
                                    ]

        elif self.semantic_problem_type == "diet1":
            possible_semantic_vars = ["grams of protein", "grams of carbohydrates", "grams of fat", "grams of fiber",
                                    "milligrams of calcium", "milligrams of iron", "milligrams of magnesium", "milligrams of potassium",
                                    "milligrams of zinc", "milligrams of vitamin A", "milligrams of vitamin B1", "milligrams of vitamin B2",
                                    "milligrams of vitamin B3", "milligrams of vitamin B4", "milligrams of vitamin B5", "milligrams of vitamin B6",
                                    "milligrams of vitamin B7", "milligrams of vitamin B9", "milligrams of vitamin B12", "milligrams of vitamin C",
                                    "milligrams of vitamin D", "milligrams of vitamin E", "milligrams of vitamin K"
                                    ]
            
        selected_vars = random.sample(possible_semantic_vars, self.num_vars)
        tokenized_vars = {var: var.split() for var in selected_vars}
        # Count token occurrences for each token
        token_counts = defaultdict(int)
        for tokens in tokenized_vars.values():
            for token in tokens:
                token_counts[token] += 1

        # Build sym vars
        sym_vars = dict()
        for i in range(self.num_vars):
            var = f"x{i}"
            varname = selected_vars[i]
            vartokens = tokenized_vars[varname]
            common_objects = [token for token in vartokens if token_counts[token] > 1]  # all repeated tokens
            differentiators = [token for token in vartokens if token not in common_objects]  # all non-repeated tokens
            # Only keep differentiators if there's at least one common object
            if not common_objects:
                differentiators = []

            item_dict = {
                'full_phrase': varname,
                'differentiators': differentiators,
                'common_objects': common_objects,
                'target': None
            }
            sym_vars[var] = item_dict

        return sym_vars

    def get_varname(self, varstring):
        # todo: simplify
        power_map = {'2': 'squared',
                    '3': 'cubed',
                    '4': 'to the fourth power',
                    '5': 'to the fifth power',
                    '6': 'to the sixth power',
                    '7': 'to the seventh power',
                    '8': 'to the eighth power',
                    '9': 'to the ninth power',
                    '10': 'to the tenth power',
                    '11': 'to the eleventh power'
                    }
        varname = ''
        if ' == ' in varstring:
            varsplit = varstring.split(' == ')
            varname += self.sym_vars[varsplit[0]]['full_phrase']
        elif ' * ' in varstring:
            varsplit = varstring.split(' * ')
            for part in varsplit:
                if ' ** ' in part:
                    partsplit = part.split(' ** ')
                    varname += self.sym_vars[partsplit[0]]['full_phrase']
                    varname += f" {power_map[partsplit[1]]} times "
                else:
                    varname += self.sym_vars[part]['full_phrase'] + " times "
            varname = varname[:-7]
        elif ' ** ' in varstring:
            varsplit = varstring.split(' ** ')
            varname += self.sym_vars[varsplit[0]]['full_phrase']
            varname += f" {power_map[varsplit[1]]}"
        else:
            varname += self.sym_vars[varstring]['full_phrase']

        return varname

    def _gen_nl_resources_dict(self):
        """Method for producing a dictionary whoses keys are 'r1','r2',...,'rm', and whose values are a dictionary with the following key-value pair:
        ------------------------
        'description': natural-language/string substitute for the variable,
        'x{i}': corresponding value determining x{i}, typically drawn from resource matrix, but conceivably a lambda expression.
        'upper_bound': upperbound constraint, maybe null.
        """

        if self.semantic_problem_type == "office supplies":
            possible_resources = ["dollar cost", "storage space", "sustainability score", "employee satisfaction impact",
                                "usefulness rating", "weight", "workplace safety impact"
                                ]
        elif self.semantic_problem_type == "family budget":
            possible_resources = ["dollar cost", "storage space", "sustainability score", "usefulness rating",
                                "weight", "portability rating", "dollar value"
                                ]
        elif self.semantic_problem_type == "gardening":
            possible_resources = ["dollar cost", "yield", "planting space", "beauty rating",
                                "resilience index", "water need", "growth speed"
                                ]
        elif self.semantic_problem_type == "network defense":
            possible_resources = ["dollar cost", "computational load", "network latency impact", "power consumption",
                                "data confidentiality impact", "data integrity impact", "data accessibility impact", "network integrity impact",
                                "available bandwidth impact"
                                ]
        elif self.semantic_problem_type == "force structure":
            possible_resources = ["offensive capability rating", "defensive capability rating", "logistics footprint", "logistical capacity",
                                "deployment weight", "mobility rating", "fuel demand", "fun factor"
                                ]
        elif self.semantic_problem_type == "personnel":
            possible_resources = ["work quality rating", "productivity rating", "computer competence rating", "paperwork competence rating",
                                "dollar cost per hour", "likelihood to quit index", "organization score"
                                ]
        elif self.semantic_problem_type == "diet0":
            possible_resources = ["dollar cost", "grams of protein", "grams of carbohydrates", "grams of fat",
                                "grams of fiber", "milligrams of calcium", "milligrams of iron",
                                "tastiness rating", "healthiness rating", "sourness index", "umami index"
                                ]
        elif self.semantic_problem_type == "diet1":
            possible_resources = ["energy stability index", "cognitive performance index", "immune support index",
                                "muscle growth index", "digestive support index", "kidney support index",
                                "cardiovascular support index"]

        selected_resources = random.sample(possible_resources, self.num_resources)
        nl_resources_dict = dict()
        for i in range(self.num_resources):
            r = f"r{i}"
            item_dict = {
                'description': selected_resources[i],
                'upper_bound': self.resources_dict['budget_caps'][i],
            }
            for j in range(self.num_vars):
                var = f"x{j}"
                weight = self.resources_dict['weights'][j, i]
                item_dict[var] = weight
            nl_resources_dict[r] = item_dict

        return nl_resources_dict

    def _gen_resources_list(self):
        """
        Method that extracts from resources dict a list of strings describing the natural language objects.

        Generates a list of natural language expressions describing the resources consumed in constraints (used to
        bound the constraint types)
        """

        resources = []
        for val in self.nl_resources_dict.values():
            resources.append(val['description'])

        return resources

    def _gen_sym_constraints(self):
        """
        Generates a dict of symbolic variables connecting them to natural language descriptions appearing in the
        constraints.

        struct: dict[str, list[str]]
        """

        sym_constraints = dict()

        # Weights / resource matrix
        k=0
        for i in range(self.num_vars):
            varname = self.sym_vars[f'x{i}']['full_phrase']
            for j in range(self.num_resources):
                resource_name = self.resources[j]
                weight = self.resources_dict['weights'][i][j]
                if self.semantic_problem_type == "personnel":
                    employee = self.sym_vars[f'x{i}']['differentiators'][0]
                    weight_statements = [f"{employee} has a {resource_name} of {weight}. ",
                                        f"{employee}'s {resource_name} is {weight}. ",
                                        ]
                elif resource_name == "storage space":
                    units = random.choice(["square feet", "sq. ft", "ft^2"])
                    weight_statements = [f"{varname} take up {weight} square feet of storage space each. ",
                                        f"{varname} occupy {weight} {units} of storage space each. ",
                                        f"{varname} each take up {weight} {units} of storage space. ",
                                        f"{varname} each occupy {weight} {units} of storage space. ",
                                        f"{varname} each take up {weight} {units}t in storage. ",
                                        f"{varname} requre {weight} {units} in storage. ",
                                        ]
                elif resource_name == "dollar cost":
                    weight_statements = [f"{varname} cost ${weight} each. ",
                                        f"{varname} cost {weight} dollars each. ",
                                        f"{varname} each cost ${weight}. ",
                                        f"{varname} each cost {weight} dollars. ",
                                        f"{varname} are each {weight} dollars. ",
                                        f"{varname} are each ${weight}. ",
                                        f"{varname} are {weight} dollars each. ",
                                        f"{varname} are ${weight} each. ",
                                        f"{varname} are {weight} dollars. ",
                                        f"{varname} are ${weight}. ",
                                        f"${weight} is the price for {varname}"
                                        f"{weight} dollars is the price for {varname}"
                                        ]
                elif resource_name == "dollar value":
                    weight_statements = [f"{varname} are worth ${weight}. ",
                                        f"{varname} are worth ${weight} each. ",
                                        f"{varname} are worth {weight} dollars. ",
                                        f"{varname} are worth {weight} dollars each. ",
                                        f"{varname} have a value of {weight} dollars. ",
                                        f"{varname} have a value of {weight} dollars each. ",
                                        f"{varname} have a value of ${weight}. ",
                                        f"{varname} have a value of ${weight} each. ",
                                        ]
                elif resource_name == "weight":
                    units = random.choice(["pounds", "lbs"])
                    weight_statements = [f"{varname} each weigh {weight} {units}. ",
                                        f"{varname} are each {weight} {units} in weight. ",
                                        f"{varname} are {weight} {units} in weight. ",
                                        f"{varname} weigh {weight} {units}. "]
                elif resource_name in ["grams of protein", "grams of carbohydrates", "grams of fat",
                                        "grams of fiber", "milligrams of calcium", "milligrams of iron",
                                        ]:
                    weight_statements = [f"{varname} each contain {weight} {resource_name}. ",
                                        f"{varname} contain {weight} {resource_name}. ",
                                        f"There are {weight} {resource_name} in {varname}. ",
                                        ]

                elif resource_name in ["energy stability index", "cognitive performance index", "immune support index", "muscle growth index",
                                        "digestive support index", "kidney support index", "cardiovascular support index"]:
                    weight_statements = [f"{varname} each have a {resource_name} of {weight}. ",
                                        f"{varname} have a {resource_name} of {weight}. ",
                                        f"The {resource_name} of {varname} is {weight}. ",
                                        f"The {resource_name} for {varname} is {weight}. ",
                                        ]
                elif resource_name == "yield":
                    weight_statements = [f"The yield of {varname} is {weight} per season.",
                                        f"{varname} yield {weight} per season. ",
                                        f"{varname} yield {weight}. ",
                                        f"The yield of {varname} is {weight}. ",
                                        ]
                elif resource_name == "planting space":
                    units = random.choice(["square inches", "sq. in", "in^2"])
                    weight_statements = [f"The planting space required for {varname} is {weight} {units}. ",
                                        f"{varname} require {weight} {units} of planting space. ",
                                        f"{varname} require {weight} {units} for planting space. ",
                                        f"{varname} each require {weight} {units} of planting space. ",
                                        f"{varname} each require {weight} {units} of space. ",
                                        f"{varname} each need {weight} {units} planting space. ",
                                        f"{varname} each need {weight} {units} of space. ",
                                        ]
                elif resource_name == "water need":
                    weight_statements = [f"{varname} each have water need of {weight} inches of water per week. ",
                                        f"{varname} each have a water need of {weight} inches of water per week. ",
                                        f"{varname} require {weight} inches of water per week",
                                        f"{varname} each require {weight} inches of water per week. ",
                                        f"{varname} require {weight} inches of water per week each. ",
                                        f"{varname} need {weight} inches of water per week. ",
                                        f"{varname} each need {weight} inches of water per week. ",
                                        f"{varname} need {weight} inches of water per week each. ",
                                        ]
                elif resource_name == "computational load":
                    units = random.choice(["TeraFLOPS", "TFLOPs"])
                    weight_statements = [f"{varname} each contribute a computational load of {weight} {units}. ",
                                        f"{varname} contribute a computational load of {weight} {units} each. ",
                                        f"{varname} each create a computational load of {weight} {units}. ",
                                        f"{varname} create a computational load of {weight} {units} each. ",
                                        ]
                elif resource_name == "network latency impact":
                    units = random.choice(["milliseconds", "ms"])
                    weight_statements = [f"{varname} each create {weight} {units} in additional network latency. ",
                                        f"{varname} create {weight} {units} in additional network latency each. ",
                                        f"{varname} each create {weight} {units} of network latency. ",
                                        f"{varname} each cause {weight} {units} of additional network latency. ",
                                        ]
                elif resource_name == "power consumption":
                    units = random.choice(["kilowatt-hours", "kWh"])
                    weight_statements = [f"{varname} each add {weight} {units} in power consumption. ",
                                        f"{varname} add {weight} {units} in power consumption each. ",
                                        f"{varname} consume {weight} {units} of power. ",
                                        f"{varname} each require {weight} {units} of power. ",
                                        f"{varname} each require {weight} {units} in power consumption. ",
                                        ]
                elif resource_name == "available bandwidth impact":
                    units = random.choice(["bits per second", "bps"])
                    weight_statements = [f"{varname} each cost {weight} {units} in network bandwidth. ",
                                        f"{varname} cost {weight} {units} in network bandwidth each. ",
                                        f"{varname} each take up {weight} {units} in network bandwidth . ",
                                        f"{varname} take up {weight} {units} in network bandwidth each. ",
                                        ]
                elif resource_name == "logistics footprint":
                    weight_statements = [f"{varname} each have a logistics footprint of {weight} logistics units. ",
                                        f"{varname} have a logistics footprint of {weight} logistics units each. ",
                                        f"The logistics footprint of {varname} is {weight} logistics units. ",
                                        f"{varname} have a logistics footprint of {weight} logistics units. ",
                                        ]
                elif resource_name == "logistics capacity":
                    weight_statements = [f"{varname} each have a logistics capacity of {weight} logistics capacity units. ",
                                        f"{varname} have a logistics capacity of {weight} logistics capacity units each. ",
                                        f"The logistics capacity of {varname} is {weight} logistics capacity units. ",
                                        f"{varname} have a logistics capacity of {weight} logistics capacity units. ",
                                        ]
                elif resource_name == "deployment weight":
                    weight_statements = [f"{varname} each have a deployment weight of {weight} metric tons. ",
                                        f"{varname} have a deployment weight of {weight} metric tons each. "
                                        f"{varname} each weigh {weight} metric tons. ",
                                        f"{varname} weigh {weight} metric tons each. ",
                                        f"The deployment weight of {varname} is {weight} metric tons each. ",
                                        ]
                elif resource_name == "fuel demand":
                    units = random.choice(["gallons per day", "gallons/day", "gal per day", "gal/day"])
                    weight_statements = [f"{varname} each have a fuel demand of {weight} {units}. ",
                                        f"{varname} have a fuel demand of {weight} {units} each. ",
                                        f"{varname} each require {weight} {units} of fuel. ",
                                        f"{varname} require {weight} {units} of fuel each. ",
                                        ]
                else:
                    weight_statements = [f"{varname} have a {resource_name} of {weight}. ",
                                        f"The {resource_name} of {varname} is {weight}. ",
                                        f"{varname} each have a {resource_name} of {weight}. ",
                                        f"{varname} have a {resource_name} of {weight} each. ",
                                        ]

                sym_constraints[f'C{k}_w'] = random.choice(weight_statements)
                k+=1

        # Non-negativity
        k = 0
        for constraint_name, constraint in self.constraints['non_negativity'].items():
            for var in self.sym_vars.keys():
                varname = self.sym_vars[var]['full_phrase']
                if var+' >= 0' in constraint:
                    if random.random() > 0.5:  # Positive statement
                        imperatives = ["You must use ", "You must have ", "You have to use ", "You have to have ",
                                       "There must be ", "There has to be "]
                        operator_statements = ["no less than zero ", "no less than 0 ", "zero or more ", "0 or more ",
                                               "zero or greater ", "0 or greater ", "greater than or equal to zero ",
                                               "greater than or equal to 0 ", "a non-negative number of ",
                                               "no less than zero ", "no less than 0 "
                                               ]
                        sym_constraint = random.choice(imperatives) + random.choice(operator_statements) + varname + ". "
                    else:  # Negative statement
                        imperatives = ["You cannot use ", "You can't use ", "You cannot have ", "You can't have ",
                                       "There cannot be ", "There can't be ", "There must not be ", "There mustn't be ",
                                       "Do not use ", "Do not have ", "Don't use ", "Don't have "]
                        operator_statements = ["less than zero ", "fewer than zero ", "a negative number of ",
                                               "less that 0 ", "fewer than 0 ", "a negative amount of "]
                        sym_constraint = random.choice(imperatives) + random.choice(operator_statements) + varname + ". "

                    sym_constraints[f'{constraint_name}'] = sym_constraint

        # Lower Bound
        for constraint_name, constraint in self.constraints['lower_bound'].items():
            if not isinstance(constraint['bound'], str):
                bound = round(constraint['bound'], 2)
                bound = str(bound)
            if "_proportion" in constraint_name:
                lb_statement = random.choice([''])
                for vartuple in constraint['tuple_list']:
                    varname = self.get_varname(vartuple[1])
                    if random.random() > 0.75:
                        lb_statement += num2words(vartuple[0]) + f' times the number of {varname}, plus '
                    else:
                        lb_statement += str(vartuple[0]) + f' times the number of {varname}, plus '
                imperative = random.choice([' must be ', ' has to be ', ' should be '])
                lb_statement = lb_statement[:-7] + imperative
                if constraint['relation'] == '>=':
                    relation = random.choice(['greater than or equal to ', 'at least ', 'at minimum ', 'no less than '])
                elif constraint['relation'] == '<=':
                    relation = random.choice(['less than or equal to ', 'at most ', 'at maximum ', 'no more than '])
                lb_statement += relation + num2words(bound) + '. '
                pass

            else:  # For all other lower bounds
                resource_idx = int(constraint_name.split('_')[1])
                resource = self.resources[resource_idx]
                joined_vars = ''
                i=1
                # Produce joined_vars string
                if random.random() > 0.5:
                    for vartuple in constraint['tuple_list']:
                        varname = self.get_varname(vartuple[1])
                        if len(constraint['tuple_list']) - i == 0:
                            conjunction = ''
                        elif len(constraint['tuple_list']) - i == 1:
                            conjunction = ", and "
                        else:
                            conjunction = ", "
                        joined_vars += varname + conjunction
                        i+=1
                else:
                    for vartuple in constraint['tuple_list']:
                        varname = self.get_varname(vartuple[1])
                        if len(constraint['tuple_list']) - i == 0:
                            conjunction = ''
                        elif len(constraint['tuple_list']) - i > 0:
                            conjunction = " plus "
                        joined_vars += varname + conjunction
                        i+=1

                if resource == "dollar cost":
                    lb_statement = random.choice([f"You must spend at least {bound} dollars on {joined_vars}. ",
                                                  f"You must spend at least ${bound} on {joined_vars}. ",
                                                  f"You have to spend at least {bound} dollars on {joined_vars}. ",
                                                  f"You have to spend at least ${bound} on {joined_vars}. ",
                                                  f"At minimum, you must spend at least ${bound} on {joined_vars}. ",
                                                  f"At minimum, you must spend at least {bound} dollars on {joined_vars}. ",
                                                  f"{joined_vars} must cost a total of at least {bound} dollars. ",
                                                  f"{joined_vars} must cost a total of at least ${bound}. ",
                                                  f"You can spend no less than ${bound} on {joined_vars}"
                                                  f"You can spend no less than {bound} dollars on {joined_vars}"
                                                  ])
                elif resource == "dollar value":
                    lb_statement = random.choice([f"{joined_vars} must have a combined value of at least {bound} dollars. ",
                                                  f"{joined_vars} have to have a combined value of at least {bound} dollars. ",
                                                  f"{joined_vars} must have a combined value of at least ${bound}. ",
                                                  f"{joined_vars} have to have a combined value of at least ${bound}. ",
                                                  f"The combined value of {joined_vars} must be at least ${bound}. ",
                                                  f"The combined value of {joined_vars} has to be at least ${bound}. ",
                                                  f"The total value of {joined_vars} must be at least ${bound}. ",
                                                  f"The total value of {joined_vars} has to be at least ${bound}. ",
                                                  ])
                elif resource == "storage space":
                    units = random.choice(["square feet", "sq. ft", "ft^2"])
                    lb_statement = random.choice([f"You must use at least {bound} {units} of storage space on {joined_vars}. ",
                                                  f"{joined_vars} must take up at least {bound} {units} of storage space. ",
                                                  f"{joined_vars} must occupy at least {bound} {units} of storage space. ",
                                                  f"You have to take up at least {bound} {units} of storage space with {joined_vars}. ",
                                                 ])
                elif resource == "weight":
                    units = random.choice(["pounds", "lbs"])
                    lb_statement = random.choice([f"The total weight of {joined_vars} should be at least {bound} {units}. ",
                                                  f"The total weight of {joined_vars} must be at least {bound} {units}. ",
                                                  f"The combined weight of {joined_vars} should be at least {bound} {units}. ",
                                                  f"The combined weight of {joined_vars} must be at least {bound} {units}. ",
                                                 ])
                elif resource in ["grams of carbohydrates", "grams of fat", "grams of fiber",
                                  "milligrams of calcium", "milligrams of iron"
                                  ]:
                    lb_statement = random.choice([f"You must get at least {bound} {resource} of from {joined_vars}. ",
                                                  f"You need to get at least {bound} {resource} of from {joined_vars}. ",
                                                  f"At least {bound} {resource} must come from {joined_vars}. ",
                                                  ])
                elif resource in ["energy stability points", "cognitive performance points", "immune support points", "muscle growth points",
                                  "digestive support points", "kidney support points", "cardiovascular support points"]:
                    lb_statement = random.choice([f"You must get at least {bound} {resource} from {joined_vars}, total. ",
                                                  f"You must get at least {bound} {resource} total from {joined_vars}. ",
                                                  f"You must get at least {bound} {resource} from {joined_vars}.",
                                                  f"{joined_vars} must provide at least {bound} {resource}",
                                                  ])
                elif resource == "yield":
                    lb_statement = random.choice([f"{joined_vars} must yield at least {bound}. ",
                                                  f"{joined_vars} must provide a yield of at least {bound}",
                                                  f"The total yield of {joined_vars} must be at least {bound}",
                                                  f"The total yield from {joined_vars} must be at least {bound}",
                                                  f"The total yield from {joined_vars} combined must be at least {bound}",
                                                  f"The total yield from {joined_vars} combined must be at least {bound}",
                                                 ])
                elif resource == "planting space":
                    units = random.choice(["square feet", "sq. ft", "ft^2"])
                    lb_statement = random.choice([f"You must use at least {bound} {units} of planting space on {joined_vars}. ",
                                                  f"{joined_vars} must take up at least {bound} {units} of planting space. ",
                                                  f"{joined_vars} must occupy at least {bound} {units} of planting space. ",
                                                  f"You have to take up at least {bound} {units} of planting space with {joined_vars}. ",
                                                  f"The total planting space occupied by {joined_vars} must be {bound} {units} or more. ",
                                                  f"The total planting space occupied by {joined_vars} must be at least {bound} {units}. ",
                                                  ])
                elif resource == "water need":
                    lb_statement = random.choice([f"The total water need of {joined_vars} must be at least {bound} inches per week. ",
                                                  f"The total water need of {joined_vars} must be minimum {bound} inches per week. ",
                                                  f"The total water need of {joined_vars} should be at least {bound} inches per week. ",
                                                  f"The number of {joined_vars} should contribute a water need of at least {bound} inches per week. ",
                                                  ])
                elif resource == "computational load":
                    units = random.choice(["TeraFLOPS", "TFLOPs"])
                    lb_statement = random.choice([f"{joined_vars} should contribute a total computational load of at least {bound} {units}. ",
                                                  f"{joined_vars} need to contribute a total computational load of at least {bound} {units}. ",
                                                  f"{joined_vars} have to contribute a total computational load of at least {bound} {units}. ",
                                                  f"The total computational load from {joined_vars} should be at least {bound} {units}. ",
                                                  f"{joined_vars} should altogether contribute be at least {bound} {units}. ",
                                                  f"The minimum computational load from {joined_vars} needs to be {bound} {units}. ",
                                                  ])
                elif resource == "network latency impact":
                    units = random.choice(["milliseconds", "ms"])
                    lb_statement = random.choice([f"{joined_vars} must contribute a minimum combined {bound} {units} of network latency. ",
                                                  f"{joined_vars} should contribute at least {bound} {units} in network latency. ",
                                                  f"The network latency from {joined_vars} must be at least {bound} {units}. ",
                                                  f"The network latency from {joined_vars} must be {bound} {units} at least. ",
                                                  f"The network latency from {joined_vars} should be at least {bound} {units}. ",
                                                  f"The network latency from {joined_vars} should be {bound} {units} at least. ",
                                                  ""])
                elif resource == "power consumption":
                    units = random.choice(["kilowatt-hours", "kWh"])
                    lb_statement = random.choice([f"{joined_vars} must have a power consumption of at least {bound} {units}. ",
                                                  f"{joined_vars} must contribute at least {bound} {units} of power consumption. ",
                                                  f"The power consumption of {joined_vars} must be at least {bound} {units}. ",
                                                  f"The power consumption of {joined_vars} must be {bound} {units} at least. ",
                                                  f"The minimum power consumption from {joined_vars} must be {bound} {units}. ",
                                                  ])
                elif resource == "available bandwidth impact":
                    units = random.choice(["bits per second", "bps"])
                    lb_statement = random.choice([f"The minimum bandwith used by {joined_vars} must be {bound} {units}. ",
                                                  f"{joined_vars} must use at least {bound} {units} of available bandwidth. ",
                                                  f"The minimum bandwidth used by {joined_vars} must be {bound} {units}. ",
                                                  f"{joined_vars} must use at least {bound} {units} of bandwidth. ",
                                                  ])
                elif resource == "logistics footprint":
                    lb_statement = random.choice([f"The total logistics footprint of {joined_vars} must be at least {bound} logistics units. ",
                                                  f"The total logistics footprint of {joined_vars} in logistics units must be at least {bound}. ",
                                                  f"The minimum logistics footprint of {joined_vars} is {bound} logistics units. ",
                                                  f"The minimum logistics footprint from {joined_vars} is {bound} logistics units. ",
                                                  ])
                elif resource == "logistics capacity":
                    lb_statement = random.choice([f"The total logistics capacity of {joined_vars} must be at least {bound} logistics capacity units. ",
                                                  f"The total logistics capacity of {joined_vars} in logistics capacity units must be at least {bound}. ",
                                                  f"The minimum logistics capacity of {joined_vars} is {bound} logistics capacity units. ",
                                                  f"The minimum logistics capacity from {joined_vars} is {bound} logistics capacity units. ",
                                                  ])
                elif resource == "deployment weight":
                    lb_statement = random.choice([f"{joined_vars} must have a deployment weight of at least {bound} metric tons. ",
                                                  f"{joined_vars} must have a total combined deployment weight of at least {bound} metric tons. ",
                                                  f"The total deployment weight of {joined_vars} must be at least {bound} metric tons. ",
                                                  f"The minimum deployment weight for {joined_vars} is {bound} metric tones. ",
                                                  ])
                elif resource == "fuel demand":
                    units = random.choice(["gallons per day", "gallons/day", "gal per day", "gal/day"])
                    lb_statement = random.choice([f"The total fuel demand of {joined_vars} must be at least {bound} {units}. ",
                                                  f"The total fuel demand of {joined_vars} must be {bound} {units}, at minimum. ",
                                                  f"The total fuel demand of {joined_vars} must be at minimum {bound} {units}. ",
                                                  f"The total fuel demand of {joined_vars} has to be at least {bound} {units}. ",
                                                  f"{joined_vars} must have a total fuel demand of at least {bound} {units}. ",
                                                  f"{joined_vars} has to have a total fuel demand of at least {bound} {units}. ",
                                                  f"{joined_vars} must contribute a total fuel demand of at least {bound} {units}. ",
                                                  ])
                else:
                    lb_statement = f"The total combined {resource} from "
                    lb_statement += f"{joined_vars} "
                    imperative = random.choice(["must be ", "has to be ", "should be "])
                    lb_statement += imperative
                    if constraint['relation'] == '>=':
                        if random.random() > 0.5:
                            operator = random.choice(['greater than or equal to ', 'equal to or greater than ',
                                                      'as much or more than ', 'at least ', 'at minimum ', 'no less than '])
                            lb_statement += operator + bound + ". "
                        else:
                            operator = random.choice([' or more. ', ' at a minimum. ', ' at minimum. '])
                            lb_statement += bound + operator
                    elif constraint['relation'] == '<=':
                        lb_statement = "[ERROR] LESS THAN STATEMENT IN LOWER BOUND"

            sym_constraints[f'{constraint_name}'] = lb_statement

        # Upper Bound
        for constraint_name, constraint in self.constraints['upper_bound'].items():
            # if "_budget" in constraint_name:
            
            if not isinstance(constraint['bound'], str):
                bound = round(constraint['bound'], 2)
                bound = str(bound)
            resource_idx = int(constraint_name.split('_')[1])
            resource = self.resources[resource_idx]
            joined_vars = ''
            i=1
            if random.random() > 0.5:
                for vartuple in constraint['tuple_list']:
                    varname = self.get_varname(vartuple[1])
                    if len(constraint['tuple_list']) - i == 0:
                        conjunction = ''
                    elif len(constraint['tuple_list']) - i == 1:
                        conjunction = random.choice([", and ", " and "])
                    else:
                        conjunction = ", "
                    joined_vars += varname + conjunction
                    i+=1
            else:
                for vartuple in constraint['tuple_list']:
                    varname = self.get_varname(vartuple[1])
                    if len(constraint['tuple_list']) - i == 0:
                        conjunction = ''
                    elif len(constraint['tuple_list']) - i > 0:
                        conjunction = " plus "
                    joined_vars += varname + conjunction
                    i+=1

            if resource == "dollar cost":
                ub_statement = random.choice([f"You must spend no more than {bound} dollars on {joined_vars}. ",
                                            f"You must spend no more than ${bound} on {joined_vars}. ",
                                            f"You must spend no greater than ${bound} on {joined_vars}. ",
                                            f"You must spend no greater than {bound} dollars on {joined_vars}. ",
                                            f"You have to spend no more than {bound} dollars on {joined_vars}. ",
                                            f"You have to spend no more than ${bound} on {joined_vars}. ",
                                            f"You have to spend no greater than {bound} dollars on {joined_vars}. ",
                                            f"You have to spend no greater than ${bound} on {joined_vars}. ",
                                            f"At maximum, you can spend ${bound} on {joined_vars}. ",
                                            f"At maximum, you can spend {bound} dollars on {joined_vars}. ",
                                            f"{joined_vars} must cost a total of at most {bound} dollars. ",
                                            f"{joined_vars} must cost a total of at most ${bound}. ",
                                            f"You can spend a maximum of ${bound} on {joined_vars}. ",
                                            f"You can spend a maximum of {bound} dollars on {joined_vars}. ",
                                            ])
            elif resource == "dollar value":
                ub_statement = random.choice([f"{joined_vars} must have a combined value of no more than {bound} dollars. ",
                                              f"{joined_vars} have to have a combined value of no more than {bound} dollars. ",
                                              f"{joined_vars} must have a combined value of no greater than ${bound}. ",
                                              f"{joined_vars} have to have a combined value of no greater than ${bound}. ",
                                              f"The combined value of {joined_vars} must be at most ${bound}. ",
                                              f"The combined value of {joined_vars} must be no greater than ${bound}. ",
                                              f"The combined value of {joined_vars} must be no more than ${bound}. ",
                                              f"The combined value of {joined_vars} has to be at most ${bound}. ",
                                              f"The combined value of {joined_vars} has to be no greater than ${bound}. ",
                                              f"The combined value of {joined_vars} has to be no more than ${bound}. ",
                                              f"The total value of {joined_vars} must be at most ${bound}. ",
                                              f"The total value of {joined_vars} must be no greater than ${bound}. ",
                                              f"The total value of {joined_vars} has to be at most ${bound}. ",
                                              f"The total value of {joined_vars} has to be no greater than ${bound}. ",
                                              ])
            elif resource == "storage space":
                units = random.choice(["square feet", "sq. ft", "ft^2"])
                ub_statement = random.choice([f"You must use at most {bound} {units} of storage space on {joined_vars}. ",
                                              f"{joined_vars} must take up no more than {bound} {units} of storage space. ",
                                              f"{joined_vars} must occupy no more than {bound} {units} of storage space. ",
                                              f"You have to take up at most {bound} {units} of storage space with {joined_vars}. ",
                                              f"A maximum of {bound} {units} of storage space can be used for {joined_vars}. ",
                                              f"Up to {bound} {units} of storage space can be used for {joined_vars}. ",
                                              f"You're not allowed to use more than {bound} {units} of storage space for {joined_vars}. ",
                                              ])
            elif resource == "weight":
                units = random.choice(["pounds", "lbs"])
                ub_statement = random.choice([f"The total weight of {joined_vars} should be no more than {bound} {units}. ",
                                              f"The total weight of {joined_vars} must be no more than {bound} {units}. ",
                                              f"The combined weight of {joined_vars} should be no more than {bound} {units}. ",
                                              f"The combined weight of {joined_vars} must be no more than {bound} {units}. ",
                                              f"{joined_vars} must weigh no more than {bound} {units}, altogether. ",
                                              f"{joined_vars} must weigh no more than {bound} {units}, total. ",
                                              f"{joined_vars} should not weigh more than {bound} {units}, altogether. ",
                                              f"{joined_vars} should not weigh more than {bound} {units}, total. ",
                                              f"{joined_vars} can't weigh more than {bound} {units}, altogether. ",
                                              f"{joined_vars} can't weigh more than {bound} {units}, total. ",
                                              ])
            elif resource in ["grams of carbohydrates", "grams of fat", "grams of fiber",
                              "milligrams of calcium", "milligrams of iron"
                              ]:
                ub_statement = random.choice([f"You must get at most {bound} {resource} of from {joined_vars}. ",
                                              f"You must get no more than {bound} {resource} of from {joined_vars}. ",
                                              f"You need to get no more than {bound} {resource} of from {joined_vars}. ",
                                              f"At most {bound} {resource} can come from {joined_vars}. ",
                                              f"You cannot get more than {bound} {resource} from {joined_vars}. ",
                                              f"You can get up to {bound} {resource} from {joined_vars}. ",
                                              ])
            elif resource in ["energy stability points", "cognitive performance points", "immune support points", "muscle growth points",
                              "digestive support points", "kidney support points", "cardiovascular support points"]:
                ub_statement = random.choice([f"You must get at most {bound} {resource} from {joined_vars}, total. ",
                                              f"You must get at most {bound} {resource} total from {joined_vars}. ",
                                              f"You must get at most {bound} {resource} from {joined_vars}.",
                                              f"{joined_vars} must provide at most {bound} {resource}",
                                              f"You can get up to a maximum of {bound} {resource} from {joined_vars}. ",
                                              f"You can get up to {bound} {resource} from {joined_vars}. ",
                                              ])
            elif resource == "yield":
                ub_statement = random.choice([f"{joined_vars} must yield no more than {bound}. ",
                                              f"{joined_vars} must provide a yield of no more than {bound}. ",
                                              f"The total yield of {joined_vars} must be at most {bound}. ",
                                              f"The total yield from {joined_vars} must be at most {bound}. ",
                                              f"The total yield from {joined_vars} combined must be at most {bound}. ",
                                              f"The total yield from {joined_vars} combined must be no more than {bound}. "
                                              ])
            elif resource == "planting space":
                units = random.choice(["square feet", "sq. ft", "ft^2"])
                ub_statement = random.choice([f"You must use no more than {bound} {units} of planting space on {joined_vars}. ",
                                              f"{joined_vars} must take up at most {bound} {units} of planting space. ",
                                              f"{joined_vars} must occupy at most {bound} {units} of planting space. ",
                                              f"You have to take up no more than {bound} {units} of planting space with {joined_vars}. ",
                                              f"The total planting space occupied by {joined_vars} must be {bound} {units} or less. ",
                                              f"The total planting space occupied by {joined_vars} must be at most {bound} {units}. ",
                                              f"The total planting space occupied by {joined_vars} must be equal to or less than {bound} {units}. ",
                                              ])
            elif resource == "water need":
                ub_statement = random.choice([f"The total water need of {joined_vars} must be no more than {bound} inches per week. ",
                                              f"The total water need of {joined_vars} must be maximum {bound} inches per week. ",
                                              f"The total water need of {joined_vars} should be no more than {bound} inches per week. ",
                                              f"The number of {joined_vars} should contribute a water need of at most {bound} inches per week. ",
                                              f"{bound} inches per week is the maximum water available for {joined_vars}. ",
                                              f"The maximum water available for {joined_vars} is {bound} inches per week. ",
                                              ])
            elif resource == "computational load":
                units = random.choice(["TeraFLOPS", "TFLOPs"])
                ub_statement = random.choice([f"{joined_vars} should contribute a total computational load of no more than {bound} {units}. ",
                                              f"{joined_vars} need to contribute a total computational load of no more than {bound} {units}. ",
                                              f"{joined_vars} have to contribute a total computational load of no more than {bound} {units}. ",
                                              f"The total computational load from {joined_vars} should be at most {bound} {units}. ",
                                              f"{joined_vars} should altogether contribute be at most {bound} {units}. ",
                                              f"The maximum computational load from {joined_vars} needs to be {bound} {units}. ",
                                              f"The maximum computational load from {joined_vars} is {bound} {units}. ",
                                              ])
            elif resource == "network latency impact":
                units = random.choice(["milliseconds", "ms"])
                ub_statement = random.choice([f"{joined_vars} must contribute a maximum combined {bound} {units} of network latency. ",
                                              f"{joined_vars} should contribute no more than {bound} {units} in network latency. ",
                                              f"The network latency from {joined_vars} must be at most {bound} {units}. ",
                                              f"The network latency from {joined_vars} must be no more than {bound} {units}. ",
                                              f"The network latency from {joined_vars} must be {bound} {units} at most. ",
                                              f"The network latency from {joined_vars} should be at most {bound} {units}. ",
                                              f"The network latency from {joined_vars} should be no more than {bound} {units}. ",
                                              f"The network latency from {joined_vars} should be {bound} {units} at most. ",
                                              ""])
            elif resource == "power consumption":
                units = random.choice(["kilowatt-hours", "kWh"])
                ub_statement = random.choice([f"{joined_vars} must have a power consumption of at most {bound} {units}. ",
                                              f"{joined_vars} must contribute no more than {bound} {units} of power consumption. ",
                                              f"The power consumption of {joined_vars} must be no more than {bound} {units}. ",
                                              f"The power consumption of {joined_vars} must be {bound} {units} or less. ",
                                              f"The maximum power consumption from {joined_vars} must be {bound} {units}. ",
                                              f"{joined_vars} should consume a maximum of {bound} {units} of power. ",
                                              f"{joined_vars} should consume no more than {bound} {units} of power. ",
                                              f"{joined_vars} should consume no greater than {bound} {units} of power. ",
                                              ])
            elif resource == "available bandwidth impact":
                units = random.choice(["bits per second", "bps"])
                ub_statement = random.choice([f"The maximum available bandwith contributed by {joined_vars} must be {bound} {units}. ",
                                              f"{joined_vars} must contribute no more than {bound} {units} of available bandwidth. ",
                                              f"{joined_vars} must contribute no greater than {bound} {units} of available bandwidth. ",
                                              f"The maximum available bandwidth from {joined_vars} must be {bound} {units}. ",
                                              ])
            elif resource == "logistics footprint":
                ub_statement = random.choice([f"The total logistics footprint of {joined_vars} must be at most {bound} logistics units. ",
                                              f"The total logistics footprint of {joined_vars} in logistics units must be no more than {bound}. ",
                                              f"The total logistics footprint of {joined_vars} in logistics units must be no greater than {bound}. ",
                                              f"The total logistics footprint of {joined_vars} in logistics units should be no more than {bound}. ",
                                              f"The total logistics footprint of {joined_vars} in logistics units should be no greater than {bound}. ",
                                              f"The maximum logistics footprint of {joined_vars} is {bound} logistics units. ",
                                              f"The maximum logistics footprint from {joined_vars} is {bound} logistics units. "])
            elif resource == "logistics capacity":
                ub_statement = random.choice([f"The total logistics capacity of {joined_vars} must be no greater than {bound} logistics capacity units. ",
                                              f"The total logistics capacity of {joined_vars} in logistics capacity units must be at most {bound}. ",
                                              f"The maximum logistics capacity of {joined_vars} is {bound} logistics capacity units. ",
                                              f"The maximum logistics capacity from {joined_vars} is {bound} logistics capacity units. "])
            elif resource == "deployment weight":
                ub_statement = random.choice([f"{joined_vars} must have a combined deployment weight of at most {bound} metric tons. ",
                                              f"{joined_vars} must have a combined deployment weight of no more than {bound} metric tons. ",
                                              f"{joined_vars} must have a total combined deployment weight of at most {bound} metric tons. ",
                                              f"The total deployment weight of {joined_vars} must be no greater than {bound} metric tons. ",
                                              f"The maximum deployment weight for {joined_vars} is {bound} metric tones. "])
            elif resource == "fuel demand":
                units = random.choice(["gallons per day", "gallons/day", "gal per day", "gal/day"])
                ub_statement = random.choice([f"The total fuel demand of {joined_vars} must be at most {bound} {units}. ",
                                              f"The total fuel demand of {joined_vars} must be {bound} {units}, at maximum. ",
                                              f"The total fuel demand of {joined_vars} must be at maximum {bound} {units}. ",
                                              f"The total fuel demand of {joined_vars} has to be no greater than {bound} {units}. ",
                                              f"The total fuel demand of {joined_vars} has to be no more than {bound} {units}. ",
                                              f"{joined_vars} must have a total fuel demand of no more than {bound} {units}. ",
                                              f"{joined_vars} has to have a total fuel demand of at most {bound} {units}. ",
                                              f"{joined_vars} must contribute a total fuel demand of at most {bound} {units}. ",
                                              f"{joined_vars} must contribute no more than {bound} {units} total fuel demand. ",
                                              ])
            else:
                ub_statement = f"The total combined {resource} from "
                ub_statement += f"{joined_vars} "
                imperative = random.choice(["must be ", "has to be ", "should be "])
                ub_statement += imperative
                if constraint['relation'] == '<=':
                    if random.random() > 0.5:
                        operator = random.choice(["less than or equal to ", "equal to or less than ",
                                                  "as much or less than ", "at most ", "at maximum ", " no more than "])
                        ub_statement += operator + bound + ". "
                    else:
                        operator = random.choice([' or less. ', ' at a maximum. ', ' at maximum. '])
                        ub_statement += bound + operator
                elif constraint['relation'] == '<=':
                    ub_statement = "[ERROR] LESS THAN STATEMENT IN LOWER BOUND"

            sym_constraints[f'{constraint_name}'] = ub_statement

        # Integer Types
        for constraint_name, constraint in self.constraints['int_types'].items():
            varname = self.get_varname(constraint['full'])
            if constraint['bound'] == "int":
                if random.random() > 0.33:
                    imperatives = ["You must use ", "You have to use ", "You must have ", "You have to have ",
                                   "There must be ", "There has to be ", "You are restricted to ",
                                   "You're restricted to ", "You are limited to ", "You're limited to "
                                   ]
                    int_statements = ["a whole number of ", "a whole number amount of ", "an integer number of ",
                                      "an integer amount of ", "a non-fractional amount of ",
                                      "a nonfractional number of "
                                      ]
                    sym_constraints[f'{constraint_name}'] = random.choice(imperatives) + random.choice(int_statements) + varname + ". "
                elif random.random() > 0.66:
                    imperatives = ["You cannot use ", "You cannot have ", "There cannot be ", "There can't be ",
                                   "There must not be ", "There mustn't be ", "Do not use ", "Do not have ",
                                   "Don't use ", "Don't have ", "You are not allowed to use ",
                                   "You are not allowed to have "]
                    int_statements = ["a fractional number of ", "a fractional amount of ",
                                      "a non-integer number of ", "a non-integer amount of ",
                                      "a non-whole number of ",
                                      ]
                    sym_constraints[f'{constraint_name}'] = random.choice(imperatives) + random.choice(int_statements) + varname + ". "
                else:
                    int_statements = [f"The number of {varname} has to be a whole number. ", f"The amount of {varname} has to be a whole number. ",
                                      f"The number of {varname} must be a whole number. ", f"The amount of {varname} must be a whole number. ",
                                      f"A whole number of {varname} must be used. ", f"An integer number of {varname} must be used. ",
                                      f"An integer number of {varname} has to be used. ", f"The number of {varname} cannot be a fraction. ",
                                      ]
                    sym_constraints[f'{constraint_name}'] = random.choice(int_statements)

            elif constraint['bound'] == "float":
                if random.random() > 0.33:
                    imperatives = ["You can use ", "You can have ", "You may use ", "You may have ",
                                   "You are allowed to use ", "You are allowed to have ", "There can be ",
                                   "There might be ", "There may be "]
                    int_statements = ["a fractional number of ", "a fractional amount of ",
                                      "a non-integer number of ", "a non-integer amount of ",
                                      "a non-whole number of ",
                                      ]
                    sym_constraints[f'{constraint_name}'] = random.choice(imperatives) + random.choice(int_statements) + varname + ". "
                elif random.random() > 0.66:
                    imperatives = ["You don't have to use ", "You do not have to use ", "You don't have to have ",
                                   "You do not have to have ", "There doesn't have to be ",
                                   "There does not have to be ", "You are not restricted to "]
                    int_statements = ["a whole number of ", "a whole number amount of ", "an integer number of ",
                                      "an integer amount of ", "a non-fractional amount of ",
                                      "a nonfractional number of ",
                                      ]
                    sym_constraints[f'{constraint_name}'] = random.choice(imperatives) + random.choice(int_statements) + varname + ". "
                else:
                    int_statements = [f"The number of {varname} does not have to be whole. ",
                                      f"The number of {varname} does not have to be a whole number. ",
                                      f"The number of {varname} does not have to be an integer. ",
                                      f"The amount of {varname} does not have to be whole. ",
                                      f"The amount of {varname} does not have to be a whole number. ",
                                      f"The amount of {varname} does not have to be an integer. ",
                                      f"The quantity of {varname} does not have to be whole. ",
                                      f"The quantity of {varname} does not have to be a whole number. ",
                                      f"The quantity of {varname} does not have to be an integer. ",
                                      f"There does not have to be a whole number amount of {varname}. ",
                                      f"There does not have to be a whole number of {varname}. ",
                                      f"There does not have to be an integer number of {varname}. ",
                                      f"There does not have to be an integer amount of {varname}. ",
                                      f"The number of {varname} can be a float. ",
                                      f"The number of {varname} can be a decimal. ",
                                      f"There can be a fractional number of {varname}. ",
                                      ]
                    sym_constraints[f'{constraint_name}'] = random.choice(int_statements)

        # Total Binder
        # sym_constraints['total_binder'] = f'{self.constraints['total_binder']}'
        # print(self.constraints['total_binder'])

        return sym_constraints
    
    def _gen_problem_constraints(self):
        problem_constraints = []
        for constraint in self.sym_constraints.values():
            problem_constraints.append(constraint)
        return problem_constraints
    
    def _gen_objective_function_statement(self):

        objective_function_statement = random.choice([f'Your overall goal is to {self.goal} ',
                                                      f'Your goal is to {self.goal} ',
                                                      f'You need to {self.goal} ',
                                                      f'You need to {self.goal} the value of ',
                                                      f'We need to {self.goal} ',
                                                      f'We need to {self.goal} the value of ',
                                                      f'Altogether, we want to {self.goal} ',
                                                      ])

        for vartuple in self.objective_function_data['term_list']:
            if not isinstance(vartuple[0], str):
                coef = round(vartuple[0], 2)
                coef = str(coef)
            operator = random.choice([' times the number of ', ' multiplied by the number of ', ' times the amount of ',
                                      ' multiplied by the amount of ', ' times the quantity of ', ' multiplied by the quantity of ',
                                      ' times the total number of ', ' times the quantity of '])
            varname = self.get_varname(vartuple[1])
            trailing_operator = random.choice([" plus ", " added to "])
            objective_function_statement += coef + operator + varname + trailing_operator
        objective_function_statement = objective_function_statement[:-len(trailing_operator)] + '. '

        return objective_function_statement
    
    def _gen_problem_statement(self):
        if self.semantic_problem_type == "office supplies":
            problem_statement = random.choice(["You are responsible for managing the supplies budget for an office. ",
                                               "I need you to help balance a budget for office supplies. ",
                                               "You will take the roll of an office manager who has to balance the budget for supplies in an office. ",
                                               "Help me balance my office supplies budget using the following criteria. "])
            # Add sym vars
            problem_statement += random.choice(["The office supplies being purchased are: ",
                                                "The following supplies are available for purchase: "
                                                "You need to balance purchases of the following supplies: "])
            i = 1
            for var in self.problem_variables:
                if i < self.num_vars-2:
                    problem_statement += var + ", "
                elif i == self.num_vars-1:
                    problem_statement += var + random.choice([" and ", ", and "])
                elif i == self.num_vars:
                    problem_statement += var + ". "
                i+=1

            # Add resources
            problem_statement += random.choice(["We need to optimize around the following resources: ",
                                                "We have to balance per all of the following metrics: ",
                                                "You will need to optimize around "])
            i = 1
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "family budget":
            problem_statement = random.choice(["You need to balance the family budget for my household. ",
                                               "I need you to help me balance my family budget. ",
                                               "I am a head of household determining the monthly family budget. I need your help. ",
                                               "I\'m trying to balance my family budget and I need help. ",
                                               "You will take the roll of a financial planner and help balance my family budget. ",
                                               "I need someone who is good with money to help me balance my family budget. Please help me, my family is dying. "])
            # Add sym vars
            if self.num_vars == 1:
                problem_statement += random.choice(["The only thing I want is ", "All we need is ", "I just need ", "We just need "])
                problem_statement += self.resources[0] + ". "
            else:
                problem_statement += random.choice(["The household supplies that we need to balance are: ",
                                                    "The supplies we need to budget for are "
                                                    "I have to buy the right number of ",
                                                    "These are all of the supplies we need to budget for: ",
                                                    "I need to budget for ",
                                                    "We desperately need ",
                                                    "My ungrateful son wants "])
                i = 1
                for var in self.problem_variables:
                    if i < self.num_vars-2:
                        problem_statement += var + ", "
                    elif i == self.num_vars-1:
                        problem_statement += var + random.choice([" and ", ", and "])
                    elif i == self.num_vars:
                        problem_statement += var + ". "
                    i+=1

            # Add sym resources
            if self.num_resources == 1:
                problem_statement += random.choice(["The only thing that matters is ", "All I care about is "])
                problem_statement += self.resources[0] + ". "
            else:
                problem_statement += random.choice(["We need to balance our budget around the following metrics: ",
                                                    "I need help budgeting around ",
                                                    "I can't figure out how to optimize ",
                                                    "Tell me how to balance ",
                                                    "I need to balance ",
                                                    "My wife is going to divorce me if I don't balance ",
                                                    "It's all over if I don't find the right "])
                i = 1
                for res in self.resources:
                    if i < self.num_resources-2:
                        problem_statement += res + ", "
                    elif i == self.num_resources-1:
                        problem_statement += res + random.choice([" and ", ", and "])
                    elif i == self.num_resources:
                        problem_statement += res + ". "
                    i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                if random.random() < 0.15:
                    problem_statement += random.choice(["So, ", "I think ", "It is critical that ", "Somehow "
                                                        "The problem is, ", "Also, ", "Oh god, "])
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "diet0":
            problem_statement = random.choice(["I need you to help me develop a diet plan that provides me the correct balance of nutrients and other metrics. ",
                                               "Help me refine my diet plan to provide the right balance of nutrients and other metrics. ",
                                               "I need help determining a diet plan that balances nutrients and other metrics correctly. ",
                                               "My diet is unbalanced, and I need to find a way to adjust my intake of several foods to get the right mix of nutrients and other metrics. ",
                                               "I need to optimize my nutrient intake. "])

            # Add sym vars
            problem_statement += random.choice(["These are the foods that I need to adjust my intake of: ",
                                                "These are all of the foods that need to be balanced in my diet: ",
                                                "These are the foods from which I need to get all of my nutrients: ",
                                                "You need to find the right intake of ",
                                                "I eat ",
                                                "The only things I eat are ",
                                                "All I eat are ",
                                                "I only eat "
                                                "My diet must consist entirely of "])
            i = 0
            for var in self.problem_variables:
                if i < self.num_vars-2:
                    problem_statement += var + ", "
                elif i == self.num_vars-1:
                    problem_statement += var + random.choice([" and ", ", and "])
                elif i == self.num_vars:
                    problem_statement += var + ". "
                i+=1

            # Add sym resources
            problem_statement += random.choice(["I need to get the right balance of ",
                                                "I need to optimize my intake of ",
                                                "I need to balance my ",
                                                "I care about optimizing "])
            i = 1
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "diet1":
            problem_statement = random.choice(["I need help adjusting my macronutrient and vitamin intake to help balance several metrics. ",
                                               "Help me adjust my macronutrient and vitamin intake to get the right balance of several metrics. ",
                                               "I need to change my macronutrient intake to satisfy several metrics. Please help me balance my intake. ",
                                               "Please help me balance my nutrient intake. "])
            # Add sym vars
            problem_statement += random.choice(["These are the available nutrients and vitamins: ",
                                                "These are the nutrients and vitamins that need to be optimized for: ",
                                                "These are all the nutrients and vitamins that need to be considered: ",
                                                "These are the nutrients and vitamins I need to balance: ",
                                                "I need to optimize my intake of "])
            i = 0
            for var in self.problem_variables:
                if i < self.num_vars-2:
                    problem_statement += var + ", "
                elif i == self.num_vars-1:
                    problem_statement += var + random.choice([" and ", ", and "])
                elif i == self.num_vars:
                    problem_statement += var + ". "
                i+=1

            # Add sym resources
            problem_statement += random.choice(["I need to optimize for ",
                                                "I need to optimize my intake around the following: ",
                                                "These are the indices I'm trying to balance: ",
                                                "Help my improve my intake regarding "])
            i = 0
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "gardening":
            problem_statement = random.choice(["I am going to be planting a garden, but I need to plan in order to balance several variables. ",
                                               "I need help finding the right balance of pants for my garden. "
                                               "You will take the roll of a gardener balancing the number of plants to order given the following parameters. Please determine the correct number of plants to order. ",
                                               "I need help determining the number of plants to install in my new garden. Please help me. ",
                                               "You will take the roll of a landscaper determining the amount of materials necessary given the following limitations. "])
            # Add sym vars
            problem_statement += random.choice(["These are the available plants: ",
                                                "These are all the plants that need to be considered: ",
                                                "You need to find the right number of ",
                                                "These are what are being planted: ",
                                                "The available plants are "])
            i = 0
            for var in self.problem_variables:
                if i < self.num_vars-2:
                    problem_statement += var + ", "
                elif i == self.num_vars-1:
                    problem_statement += var + random.choice([" and ", ", and "])
                elif i == self.num_vars:
                    problem_statement += var + ". "
                i+=1

            # Add sym resources
            problem_statement += random.choice(["You need to optimize the garden for: ",
                                                "You need the right mix of plants to balance ",
                                                "For the whole garden, the following need to be optimized: ",
                                                "The garden needs to have the optimal "])
            i = 0
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "personnel":
            problem_statement = random.choice(["We have a staffing imbalance in our office. Please help us balance the correct work-hour allocation given the following parameters. ",
                                               "You are to take the roll of an office manager and find the right number of hours for each employee to work given the following. ",
                                               "Help me balance the necessary hours worked by our employees to satisfy the following requirements. ",
                                               "I am an office manager and I need to balance the hours worked by our employees to satisfy the following limitations. I need your help. ",
                                               "You are an office manager and you need to determine the number of hours each employee should be working to maximize office output."])

            # Add sym vars
            problem_statement += random.choice(["You need to balance ",
                                                "You need to find the right distribution of ",
                                                "You need to determine the right number of ",
                                                "You need to optimize the following: "])
            i = 0
            for var in self.problem_variables:
                if i < self.num_vars-2:
                    problem_statement += var + ", "
                elif i == self.num_vars-1:
                    problem_statement += var + random.choice([" and ", ", and "])
                elif i == self.num_vars:
                    problem_statement += var + ". "
                i+=1

            # Add sym resources
            problem_statement += random.choice(['We have our own in-house metrics for assessing our employees. ',
                                                'We have a set of metrics for assessing our employees. ',
                                                'We rate our employees using ',
                                                'Our employees are assessed using metrics determined by a highly scientific process. '])
            problem_statement += random.choice(["Hours worked by each employee needs to be adjusted to optimize: ",
                                                "You need to find the best distribution of hours worked per the following metrics: ",
                                                "You need to optimize around the following metrics for our office: ",
                                                "We have to balance "])
            i = 0
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "network defense":
            problem_statement = random.choice(["You are in charge of a Security Operations Center for a cybersecurity operation. You need to determine the correct balance of several variables to ensure optimal performance. ",
                                               "I'm running a Security Operations Center and need help balancing several variables to ensure optimal performance. Please help. ",
                                               "I need help determining the number of certain specialists to hire and certain actions to take for the continued operation of my cybersecurity center. "])
            # Add sym vars
            problem_statement += random.choice(["You need to balance ",
                                                "You need to find the right distribution of ",
                                                "You need to determine the correct number of ",
                                                "You need to optimize the totals for the following: "])
            i = 0
            for var in self.problem_variables:
                if i < self.num_vars-2:
                    problem_statement += var + ", "
                elif i == self.num_vars-1:
                    problem_statement += var + random.choice([" and ", ", and "])
                elif i == self.num_vars:
                    problem_statement += var + ". "
                i+=1

            # Add sym resources
            problem_statement += random.choice(["You must optimize around ",
                                                "You have to optimize per ",
                                                "You are have to balance ",
                                                "You have to optimize around the following resources "
                                                ])
            i = 0
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        elif self.semantic_problem_type == "force structure":
            problem_statement = random.choice(["You are a JTF commander responsible for determining the force package required to meet mission objectives. ",
                                               "I am a JTF commander responsible for determining the force package required to meet mission objectives. I need help balancing my force structure per given restrictions. ",
                                               "Your goal is to adjust a force structure to meet mission requirements within certain limitations. ",
                                               "You are to determine the optimal force structure to meet mission requirements within the given parameters. "])

            # Add sym vars
            if random.random() > 0.5:
                problem_statement += random.choice(["These are the units you have to select from: ",
                                                    "These are the available unit types: ",
                                                    "These are all of the unit types that will be deployed: ",
                                                    "You need to optimize counts of ",
                                                    "The force will consist of "])
                i = 0
                for var in self.problem_variables:
                    if i < self.num_vars-2:
                        problem_statement += var + ", "
                    elif i == self.num_vars-1:
                        problem_statement += var + random.choice([" and ", ", and "])
                    elif i == self.num_vars:
                        problem_statement += var + ". "
                    i+=1
            else:
                problem_statement += random.choice(["You need to determine how many ",
                                                    "You have to optimize how many ",
                                                    "You will need to determine how many ",
                                                    "You will need to calculate how many ",
                                                    "You need to determine the optimal number of "])
                i = 0
                for var in self.problem_variables:
                    if i < self.num_vars-2:
                        problem_statement += var + ", "
                    elif i == self.num_vars-1:
                        problem_statement += var + random.choice([" and ", ", and "])
                    elif i == self.num_vars:
                        problem_statement += var
                    i+=1
                problem_statement += random.choice([" to deploy. ",
                                                    " should be deployed. ",
                                                    " need to be deployed. ",
                                                    " to be included in the force. ",
                                                    " to include in the force. "])

            # Add sym resources
            problem_statement += random.choice(["The force must balance the following: ",
                                                "You have to find a force structure that optimizes for the following: ",
                                                "Mission success relies on finding the optimal ",
                                                "The force structure must have optimal values of ",
                                                "You are restricted by "])
            i = 0
            for res in self.resources:
                if i < self.num_resources-2:
                    problem_statement += res + ", "
                elif i == self.num_resources-1:
                    problem_statement += res + random.choice([" and ", ", and "])
                elif i == self.num_resources:
                    problem_statement += res + ". "
                i+=1

            # Add constraints
            for constraint in self.sym_constraints.values():
                if random.random() < 0.15:
                    problem_statement += random.choice(['Critically, ', 'It is critical to account, ',
                                                        'As a rule, ', 'Please note, ', 'Additionally, '])
                problem_statement += constraint

            # Add objective statement
            problem_statement += self.objective_function_statement

        return problem_statement

    def _get_problem_variables(self):
        """Returns a list of the values from the sym_vars dictionary"""
        problem_variables = []
        for var in self.sym_vars.values():
            problem_variables.append(var['full_phrase'])

        return problem_variables
    
    def to_dict(self):
        att_dict = {}
        # return dict({
        #     "num_vars": self.num_vars,
        #     "num_resources":self.num_resources,
        #     "resources_dict": self.resources_dict,
        #     "problem_type": self.problem_type,
        #     "goal":self.goal,
        #     "problem_statement": self.problem_statement,
        #     "objective_function_statement": self.objective_function_statement,
        #     "objective_function": self.objective_function_data.get('objective_function',None),
        #     "problem_weights": self.problem_weights,
        #     "sym_weights": self.sym_weights,
        #     "resources": self.resources,
        #     "sym_vars": self.sym_vars,
        #     "problem_constraints": self.problem_constraints,
        #     "problem_variables": self._get_problem_variables(),
        #     "problem_type": self.problem_type,
        #     "goal": self.goal,
        #     "sym_constraints": self.sym_constraints,
        #     "gurobi_code": self.gurobi_code,
        #     "problem_solution":self.problem_solution,
        #     })
        for key,value in self.__dict__.items():
            att_dict.__setitem__(key,value)
        return att_dict
    
    def _gen_gurobi_code(self):
        code = "from gurobipy import *\n\n"
        # Define function name
        code += "def make_and_optimize_model():\n"
        # Add model
        code += """\tm = Model("Optimization_Problem")\n"""
        # Define Variables
        for key in self.sym_vars.keys():
            code+=f"""\t{key} = m.addVar(vtype = {self._get_var_type(key)}, name = '{key}')\n"""
        # Set Objective Function
        if self.goal == "maximize":
            goal_term = "GRB.MAXIMIZE"
        else:
            goal_term = "GRB.MINIMIZE"
        code +=f"""\tm.setObjective({self.objective_function_data['objective_function']}, {goal_term})\n"""
        # Add Constraints
        #for the constraints in sym_constraints
        for ctype, cdict in self.constraints.items():
            if ctype != 'int_types' and ctype !="total_binder":
                for cname, constraint in cdict.items():
                    code+=f"""\tm.addConstr({constraint['description']})\n"""
        # Optimize Model
        code += f"""\tm.optimize()\n"""
        code += "\treturn m"
        return code
    
    def _get_var_type(self, key:str, package="gurobi"):
        if package =="gurobi":
            if self.constraints.get('int_types',dict()).get('C_t_'+str(key),dict()).get("bound",None) == "int":
                return "GRB.INTEGER"
            else:
                return "GRB.CONTINUOUS"
        else:
            return "NULL"

    def _run_gurobi_code(self):
        generated_fun = self.gurobi_code
        
        # proceed to run the gurobi code
        local_scope = {}
        try:
            exec(generated_fun, globals())
            m = make_and_optimize_model()
        except Exception as E:
            print(f"{self.problem_statement}")
            print(f"{self.gurobi_code}")
            print(E)
            raise NameError
        solution = dict()
        v_solution = dict()
        if m.status == GRB.OPTIMAL:
            for v in m.getVars():
                try:
                    v_solution.__setitem__(v.VarName, v.X)
                except AttributeError:
                    try:
                        v_solution.__setitem__(v.VarName, v.Xn)
                    except Exception as E:
                        print(f" Error {E} on input {v}")
                        v_solution.__setitem__(v.VarName,0)
            try:
                solution.__setitem__('Value',m.ObjVal)
            except AttributeError:
                try:
                    solution.__setitem__('Value',m.ObjNVal)
                except Exception as E:
                    print(f" Error {E} on input {m}")
        else:
            solution.__setitem__('Value','Infeasible')
        solution.__setitem__("Optimal solution",v_solution)
        return solution

    def save_problem(self, dir = None):
        def convert_numpy(obj):
            if isinstance(obj, np.integer):
                return int(obj)
            elif isinstance(obj, np.floating):
                return float(obj)
            elif isinstance(obj, np.ndarray):
                return obj.tolist()
            else:
                raise TypeError(f"Object of type {obj.__class__.__name__} is not JSON serializable")

        timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        if isinstance(dir,type(None)):
            dir = os.path.join(os.getcwd(), "oproblems", f'problem_{timestamp}')
        os.makedirs(dir, exist_ok=True)
        with open(os.path.join(dir, 'problem_dict.pkl'), 'w') as f:
            pickle.dump(self.to_dict(), f)
        # with open(os.path.join(dir, 'constraints.json'), 'w') as f:
        #     json.dump(self.sym_constraints, f, indent=4)
        # with open(os.path.join(dir, 'vars.json'), 'w') as f:
        #     json.dump(self.sym_vars, f, indent=4)
        # with open(os.path.join(dir, 'objective_function_data.json'), 'w') as f:
        #     json.dump(self.objective_function_data, f, indent=4, default=convert_numpy)
        # with open(os.path.join(dir, 'objective_function_statement.txt'), 'w') as f:
        #     f.write(self.objective_function_statement)
        # with open(os.path.join(dir, 'problem_description.txt'), 'w') as f:
        #     f.write(self.problem_statement)

def convert_nlp4lp_to_Problem(nlp_dict:dict)->Problem:
    p_dict = dict()
    #Generate empty values for: num_vars, resource_dict, problem_type, num_resources, problem_variables, nl_resources_dict, resources
    None_list = ['resource_dict','problem_type','num_resources', 'problem_variables','nl_resources_dict','resources',\
        'semantic_problem_type','sym_constraints','problem_constraints','objective_function_statement']
    for key in None_list:
        p_dict.__setitem__(key,None)
    p_dict.__setitem__('resources_dict',dict())
    p_dict.__setitem__("problem_statement", nlp_dict.get("document",''))
    p_dict.__setitem__('num_vars',len(nlp_dict.get('vars',0)))
    # Generate Goal
    p_dict.__setitem__('goal',nlp_dict.get('obj_declaration',dict()).get('direction',None))
    # Generate sym_vars
    sym_vars=dict()
    # selected_vars = [key for key,_ in nlp_dict["var_mention_to_first_var"].items()]
    # tokenized_vars = {var: var.split() for var in selected_vars}
    # token_counts = defaultdict(int)
    # for i in range(p_dict['num_vars']):
    #         var = f"x{i}"
    #         varname = selected_vars[i]
    #         vartokens = tokenized_vars[varname]
    #         common_objects = [token for token in vartokens if token_counts[token] > 1]  # all repeated tokens
    #         differentiators = [token for token in vartokens if token not in common_objects]  # all non-repeated tokens
    #         # Only keep differentiators if there's at least one common object
    #         if not common_objects:
    #             differentiators = []

    #         item_dict = {
    #             'full_phrase': varname,
    #             'differentiators': differentiators,
    #             'common_objects': common_objects,
    #             'target': None
    #         }
    #         sym_vars[var] = item_dict
    for i,name in enumerate(nlp_dict.get('vars',[])):
        sym_vars.__setitem__('x'+str(i),dict({"full_phrase":name,
                                            "differentiators":[],
                                            "common_objects":[],
                                            "target":None}))
    p_dict.__setitem__("sym_vars",sym_vars)        
    # Generate objective_function_data
    def get_sym_varmatching_keys(d: dict, target:str) -> list:
        target = target.lower()
        return [k for k, v in d.items() if v.get("full_phrase",'').lower() in target]
    # print(f"DEBUG: the sym vars are {p_dict["sym_vars"]}")
    term_list = []
    for term,value in nlp_dict.get("obj_declaration",dict()).get("terms",dict()).items():
        value=value.replace(',','')
        if '%' in value:
            value = f".{value.replace('%','')}"
        # KEY ISSUE HERE: the terms that appear in obj_declaration terms are not necessarily the var names due to poor nlp org
        # The solution is to use var_mention_to_first_var, and first_var_to_mentions lists
        item = None
        for key in nlp_dict.get("vars"):
            if term in nlp_dict["first_var_to_mentions"][key]:
                item = key
                break
        if isinstance(item,type(None)):
            raise KeyError 
        # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
        # if len(items)>0:
            # item = items[0]
        ft = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item)
        term_list.append((value,ft[0]))
    # print(f"DEBUG: {term_list} derived from {nlp_dict["obj_declaration"]}")
    objective_function = ""
    for term_pair in term_list:
        objective_function += f"{term_pair[0]} * {term_pair[1]} + "
    objective_function = objective_function[:-2]
    p_dict.__setitem__('objective_function_data',dict({"objective_function":objective_function,"term_list":term_list}))
    # print(f"{p_dict["objective_function_data"]}")
    # Generate constraints dict
    
    
    lb_constraints=dict()
    ub_constraints=dict()
    N = p_dict.get('num_vars')
    for n,const in enumerate(nlp_dict.get("const_declarations",dict())):
        # handle bound first
        bound = str(const.get("limit",'0'))
        bound = bound.replace(',','')
        if bound =="five":
            bound = "5"
        elif bound == "third":
            bound = f"{1/3}"
        elif bound == "seven":
            bound = "7"
        elif bound == "ten":
            bound = "10"
        elif bound == "twenty":
            bound = 20
        elif bound == "half":
            bound = f"{1/2}"
        elif bound == "fifteen":
            bound = "15"
        elif '%' in bound:
            bound = f".{bound.replace('%','')}"
        else:
            bound = str(bound) if not isinstance(bound,type(None)) else "0"
        
        # logic for type, direction, limit, operator into constraints
        if const['type'] == 'sum':
            term_list = [(1,var) for var, _ in p_dict.get("sym_vars",dict()).items()]
            relation = ""
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation ="<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation =">="
            else:
                ord_relation ="=="
            # bound = str(const.get("limit"))
            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound.replace(',','')}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
        if const['type'] == "lowerbound":
            relation = ""
            term_list = []
            # print(f"var: {const.get("var")}")
            # print(f"sym_vars: {p_dict["sym_vars"]}")
            for key in nlp_dict.get("vars"):
                if const.get("var") in nlp_dict["first_var_to_mentions"][key]:
                    item = key
                    break
            if isinstance(item,type(None)):
                raise KeyError 
            # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
            # if len(items)>0:
                # item = items[0]
            ft = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item)[0]
            # get_sym_varmatching_keys(d= p_dict.get("sym_vars",dict()),target = const.get("var",''))[0]
            term_list.append((1,ft))
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation ="<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation =">="
            else:
                ord_relation ="=="
            # bound = str(const.get("limit"))
            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
        if const['type'] == "upperbound":
            #
            relation = ""
            term_list = []
            # term_list.append((1,get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),const.get("var",''))[0]))
            # print(f"var: {const.get("var")}")
            # print(f"sym_vars: {p_dict["sym_vars"]}")
            for key in nlp_dict.get("vars"):
                if const.get("var") in nlp_dict["first_var_to_mentions"][key]:
                    item = key
                    break
            if isinstance(item,type(None)):
                raise KeyError 
            # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
            # if len(items)>0:
                # item = items[0]
            ft = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item)[0]
            # get_sym_varmatching_keys(d= p_dict.get("sym_vars",dict()),target = const.get("var",''))[0]
            term_list.append((1,ft))
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation ="<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation =">="
            else:
                ord_relation ="=="
            # bound = str(const.get("limit"))
            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
        if const["type"] == "ratio":
            #
            relation = ""
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation ="<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation =">="
            else:
                ord_relation ="=="
            total = ""
            # value = const.get("limit','1")
            # if '%' in value:
            #     value = "."+value.replace('%','')
            bound = bound.replace(',','')
            if bound =="five":
                bound = "5"
            elif bound == "third":
                bound = f"{1/3}"
            elif bound == "seven":
                bound = "7"
            elif bound == "ten":
                bound = "10"
            elif bound == "twenty":
                bound = 20
            elif bound == "half":
                bound = f"{1/2}"
            elif bound == "fifteen":
                bound = "15"
            elif '%' in bound:
                bound = f".{bound.replace('%','')}"
            total_term_list = [(bound,term) for term,_ in p_dict.get("sym_vars",dict()).items()]
            for pair in total_term_list:
                total += f"{pair[0]} * {pair[1]} + "
            total = total[:-2]
            bound = f"{total}"
            
            
            # term_list = [(1,get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),const.get("var",''))[0])]
            for key in nlp_dict.get("vars"):
                if const.get("var") in nlp_dict["first_var_to_mentions"][key]:
                    item = key
                    break
            if isinstance(item,type(None)):
                raise KeyError 
            # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
            # if len(items)>0:
                # item = items[0]
            ft = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item)[0]
            # get_sym_varmatching_keys(d= p_dict.get("sym_vars",dict()),target = const.get("var",''))[0]
            term_list=[(1,ft)]
            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound.replace(',','')}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
        if const["type"] == "xby":
            if const["param"] == "third" or const["param"] == "a third" or const["param"] == "one third" or const["param"] == "third":
                p = 1/3
            elif const["param"] == "half":
                p = 0.5
            elif const["param"] == "1.5 times":
                p = 1.5
            elif const["param"] == "twice" or const["param"] == "2 times" or const["param"] == "two times":
                p = 2
            elif const["param"] == "three" or const["param"] == "thrice" or const["param"] == "3 times" or const["param"] == "three times":
                p = 3
            elif const["param"] == "4" or const["param"] == "4 times" or const["param"] == "four times" or const["param"] == "four":
                p = 4 
            elif const["param"] == "five" or const["param"] == "5 times" or const["param"] == "5":
                p = 5
            else:
                p = float(const["param"])
            relation = ""
            term_list = []
            item1 = None
            item2 = None
            for key in nlp_dict.get("vars"):
                if const.get("x_var") in nlp_dict["first_var_to_mentions"][key]:
                    item1 = key
                    break
            if isinstance(item1,type(None)):
                raise KeyError
            
            for key in nlp_dict.get("vars"):
                if const.get("y_var") in nlp_dict["first_var_to_mentions"][key]:
                    item2 = key
                    break
            if isinstance(item2,type(None)):
                raise KeyError  
            # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
            # if len(items)>0:
                # item = items[0]
            ft1 = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item1)[0]
            ft2 = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item2)[0]
            term_list.append((1,ft1))
            term_list.append((-p,ft2))
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation = "<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation = ">="
            else:
                ord_relation = "=="
            bound = "0"
            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound.replace(',','')}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
        if const["type"] == "linear":
            #
            relation = ""
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation ="<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation =">="
            else:
                ord_relation ="=="
            # bound = str(const.get("limit"))
            term_list = []
            for term,value in const['terms'].items():
                if '%' in value:
                    value = f".{value.replace('%','')}"
                for key in nlp_dict.get("vars"):
                    if term in nlp_dict["first_var_to_mentions"][key]:
                        item = key
                        break
                if isinstance(item,type(None)):
                    raise KeyError 
                # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
                # if len(items)>0:
                    # item = items[0]
                ft = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item)[0]
                
                term_list.append((float(value),ft))

            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound.replace(',','')}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
        if const["type"] == "xy":
            #
            relation = ""
            term_list = []
            item1 = None
            item2 = None
            # term_list.append((1,get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),const.get("x_var",''))[0]))
            # term_list.append((-1,get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),const.get("y_var",''))[0]))
            for key in nlp_dict.get("vars"):
                if const.get("x_var") in nlp_dict["first_var_to_mentions"][key]:
                    item1 = key
                    break
            if isinstance(item1,type(None)):
                raise KeyError
            for key in nlp_dict.get("vars"):
                if const.get("y_var") in nlp_dict["first_var_to_mentions"][key]:
                    item2 = key
                    break
            if isinstance(item2,type(None)):
                raise KeyError  
            # items = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target=nlp_dict["first_var_to_mentions"][key])
            # if len(items)>0:
                # item = items[0]
            try:
                ft1 = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item1)[0]
                ft2 = get_sym_varmatching_keys(p_dict.get("sym_vars",dict()),target = item2)[0]
            except Exception as E:
                print(f"""For items {item1} and {item2} corresponding to {const.get("x_var")} and {const.get("y_var")} inside {p_dict.get("sym_vars")}, we raise {E}. 
                We know that nlp_dict's first var mentions are : {nlp_dict["first_var_to_mentions"]}""")
                raise AttributeError
                
            term_list.append((1,ft1))
            term_list.append((-1,ft2))
            if const.get("operator") == "LESS_OR_EQUAL":
                ord_relation ="<="
            elif const.get("operator") == "GREATER_OR_EQUAL":
                ord_relation =">="
            else:
                ord_relation ="=="
            bound = "0"
            
            for termpair in term_list:
                relation += f"{termpair[0]} * {termpair[1]} + "
            relation= relation[:-2]+f"{ord_relation} {bound.replace(',','')}"
            if ord_relation == "<=":
                lb_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
            else:
                ub_constraints.__setitem__("C_"+str(n+N),dict({
                    "description": relation,
                    "tuple_list": term_list,
                    "relation": ord_relation,
                    "bound": str(bound)
                }))
    nndict = dict()
    is_int = dict()
    for n,var in enumerate(list(p_dict["sym_vars"].keys())):
        nndict.__setitem__("C_"+str(n),dict({"description":str(var)+" >= 0",
                                                            'tuple_list':[(1,var)],
                                                            'relation':">=",
                                                            "bound":0}))
        is_int.__setitem__("C_t_"+str(n), dict({'full':f"{var} == float",'relation':'==','bound':'float'}))
    p_dict.__setitem__("constraints",dict({"non_negativity":nndict,
                                           "int_types":is_int,
                                           "lower_bound":lb_constraints,
                                           "upper_bound":ub_constraints}))
    # print(f"{p_dict["constraints"]}")
    # Generate and Return Problem
    return Problem(**p_dict)

def generate_boolean_vectors(L,N):
    """Given a list of integers L, generate the selector Boolean vectors for the monomials of degree D in descending order with a total of N true values"""
    if sum(L)>=N:
        indices = [(i,j) for i,length in enumerate(L) for j in range(length)]
        first_indices = [(0,j) for j in range(len(L))]
        chosen_first = random.choice(first_indices)
        selected = {chosen_first}
        remaining = [idx for idx in indices if idx != chosen_first]
        selected.update(random.sample(remaining, N - 1))
        selector = [[(i,j) in selected for j in range(sub)] for i,sub in enumerate(L)]
        return selector
    else:
        print(f"Error: There are {sum(L)} in {L} to {N}")
        raise Exception("Insufficient sample size.")

def generate_monomials(vars:list[str],degree:int):
    index_combinations = combinations_with_replacement(range(len(vars)),int(degree))
    monomials = []
    for combo in index_combinations:
        counter = Counter(combo)
        parts = []
        for var_idx, count in sorted(counter.items()):
            if count == 1:
                parts.append(vars[var_idx])
            else:
                parts.append(f"{vars[var_idx]} ** {count}")
        monomials.append(' * '.join(parts))
    return monomials

def generate_problems_object(n:int, path:str, as_json= False):
    # problem_dict = dict()
    for n in range(0,n):
        try:
            problem_dict = Problem().to_dict()
            if not os.path.exists(os.path.join(path,f'prob_{n}')):
                os.makedirs(os.path.join(path,f'prob_{n}'))
            if as_json:
                with open(os.path.join(path, f"prob_{n}.json"), 'w') as file:
                    json.dump(problem_dict, file, indent=4)
            else:
                with open(os.path.join(path,f'prob_{n}', 'problem_dict.pkl'), 'wb') as f:
                    pickle.dump(problem_dict, f)
        except Exception as E:
            print(f"Following error was raised: {E}")