import time
import os
import sqlite3 
import os
import re
from collections import defaultdict
from itertools import combinations
from collections import Counter

class SymInterchange():
    """ Intermediate object containing both relevant problem strings and their symbolic representation  """
    
    def __init__(self,**sym_dict):
        """We initialize a SymInterchange object with a potentially empty dictionary. The SymInterchange object has methods for 
        extracting from natural language string descriptions the following information:
        
        :id: Optional string identifying the particular SymInterchange object. 
        :problem_statement: a natural language string describing the optimization problem
        :problem_type: a string describing if the problem is one of the following : IP, MIP, or Continuous,
        :problem_variables: a list of strings describing the problem variables
        :objective_function_statement: a natural language string describing the objective function as composed of the problem variables, and problem weights
        :objective_function: a string object in reduced symbolic form relating the problem weights and sym vars
        :problem_weights: a list of strings describing the problem weights that appear in the objective function
        :sym_weights: a dictionary whose key-value pairs are symbolic vars and either a numeric value or numeric-valued function; presently deprecated
        :resources: a list of strings describing the resources that constrain the problem variables
        :sym_vars: a dictionary whose key-value pairs are either x_i and the corresponding problem_variable string, or r_i and the corresponding resource string
        :problem_constraints: a list of strings describing the natural language the constraints that appear in the problem
        :sym_constraints: a list of of strings formed by reducing the problem_constraints to their symbolic counterparts by substituting the problem_variables and resources
        with their sym_var counterparts, and natural language description of their relations with the appropriate algebraic or order-relation counterparts

        Dually, the Problem class has methods that produce consistent natural language descriptions from the corresponding symbolic
        objects. 
        """
        self.id = sym_dict.get('id',None)
        
        self.num_vars = sym_dict.get('num_vars',None) # just an integer
        self.problem_variables = sym_dict.get('problem_variables',[]) # a list of the full_phrase values forming the sym_vars objects
        self.sym_vars = sym_dict.get('sym_vars',dict()) # keys full_phrase, differentiators, common_objects, target
        
        self.num_resources = sym_dict.get('num_resources',None) # just an integer count keeping track of the different resource types -- each variable MUST have some information
        self.resources = sym_dict.get('resources',[]) # a list capturing the resources
        self.nl_resources_dict = sym_dict.get('nl_resources_dict',dict()) # dict keys : description -- the value describing the relationship betweeen variables; upperbound -- the binding value 
        self.resources_dict = sym_dict.get('resources_dict', dict()) # dict keys : weights -- function or scalar value corresponding to term ; budget_caps:  
        
        # self.semantic_problem_type = sym_dict.get('problem_type',None)
        self.goal = sym_dict.get('goal',None) # maximize or minimize
        self.problem_type = sym_dict.get('problem_type',None) # one of three values 0.0, 0.5, or 1.0 for continuous , mixed integer, integer
        self.objective_function_data = sym_dict.get('objective_function_data',dict()) # keys: objective_function : str, term_list : list of pairs (coefficient , monomial term) appearing in the objective function
        self.objective_function_statement = sym_dict.get('objective_function_statement',"") # String natural language description of objective function, substituting in variables with corresponding string in sym_vars dictionary
        
        # self.constraints = sym_dict.get('constraints',dict({"non_negativity":dict(),"lower_bound":dict(),"upper_bound":dict(),"int_types":dict()})) # each dict has a constraint corresponding to one of 4 types, with key being constraint id, and value being expression
        self.problem_constraints = sym_dict.get('problem_constraints',[]) # List summarizing problem constraints, formed by substituting and reformulating the sym_constraints -- natural language conversion
        self.sym_constraints = sym_dict.get('sym_constraints',dict()) # dict keys: constraint name: string representing  
        self.problem_statement = sym_dict.get('problem_statement',"") # String describing problem statement
        
        # self.code = sym_dict.get('code',"")
        # self.problem_solution = sym_dict.get('problem_solution',self._run_code())
        
    def __generate_sym_problem_variables__(self):
        # Tokenize and extract keywords from each variable description
        keywords_list = [re.findall(r'\b\w+\b', value.lower()) for value in self.problem_variables]

        # Identify common and differentiating words
        # all_keywords = set(word for keywords in keywords_list for word in keywords) # May use later. Unnecessary for now within present iteration of method.
        # print(f"DEBUG STATEMENT: The generated keywords list is {keywords_list}")
        if keywords_list !=[]:
            common_words = set.intersection(*map(set, keywords_list))
        else:
            common_words = set({})
        differentiators = [list(set(kw) - common_words) for kw in keywords_list]
        # Heuristic: assume the target is the first non-differentiator noun in each phrase
        targets = []
        for kw in keywords_list:
            diff = set(kw) - common_words
            tgt = [w for w in kw if w not in diff][0] if kw else None
            targets.append(tgt)

        variable_map = {}
        for i, value in enumerate(self.problem_variables):
            variable_map[f'x{i+1}'] = {
                'full_phrase': value,
                'differentiator': differentiators[i],
                'common_objects': list(common_words),
                'target': targets[i]
            }
        self.sym_vars.update(variable_map)
    
    def __generate_sym_resources__(self):
        # Tokenize and extract keywords from each variable description
        keywords_list = [re.findall(r'\b\w+\b', value.lower()) for value in self.resources]

        # Identify common and differentiating words
        all_keywords = set(word for keywords in keywords_list for word in keywords)
        common_words = set.intersection(*map(set, keywords_list))
        differentiators = [list(set(kw) - common_words) for kw in keywords_list]
        # Heuristic: assume the target is the first non-differentiator noun in each phrase
        targets = []
        for kw in keywords_list:
            diff = set(kw) - common_words
            tgt = [w for w in kw if w not in diff][0] if kw else None
            targets.append(tgt)

        variable_map = {}
        for i, value in enumerate(results):
            variable_map[f'r{i+1}'] = {
                'full_phrase': value,
                'differentiator': differentiators[i],
                'common_objects': list(common_words),
                'target': targets[i]
            }
        self.sym_vars.update(variable_map)
    
    def _generate_sym_vars(self):
        """
        Given the current problem variables and resources, convert them into symbolic variables and to the self.sym_vars 
        if not present
        """
        self.__generate_sym_problem_variables__()
        self.__generate_sym_resources__()
    
    def _sym_substitution(self,input:str)->str:
        """Method for substituting in sym vars in a constraint, reducing it to its symbolic expression"""
        substituted_text = input
        expr = None
        for var, info in self.sym_vars.items():
            for diff in info['differentiator']:
                if diff in substituted_text:
                    substituted_text = substituted_text.replace(diff,var)
                break
            # Remove all common_objects in original text 
            for common in info['common_objects']:
                if common in substituted_text:
                    substituted_text = substituted_text.replace(common,'')
        return substituted_text
    
    def _sym_substitution_objective(self):
        """Substitute in the sym_vars into the objective function"""
        substituted_text = self._sym_substitution(self.objective_function_statement)
        pass
    
    def _sym_substitution_constraint(self,input:str)->str:
        substituted_text = self._sym_substitution(input)
        # check if the constraint fixes a type, or otherwise is a min/max constraint
        
        if 'type' in substituted_text:
            pass
        elif 'min' in substituted_text | 'max' in substituted_text:
            match = re.search(r'(maximum|min(?:imum)?)\s+of\s+(\d+).*?from\s+(.+)', substituted_text)
            if not match:
                return f"# Unparsed: {text}"

            kind = match.group(1)
            bound = match.group(2)
            terms = match.group(3)

            # Extract all coefficient-variable pairs like "30 x1"
            pairs = re.findall(r'(\d+)\s*(x\d+)', terms)

            if not pairs:
                return f"# Unparsed (no terms): {substituted_text}"

            expr = ' + '.join([f"{coeff}*{var}" for coeff, var in pairs])
            op = "<=" if "max" in kind else ">="
            return f"{expr} {op} {bound}"
        else:
            return f"# Constraint Cannot be Parsed: {substituted_text}"
        
    def _gen_sym_constraints(self):
        """Given the current problem variables, resources, symbolic variables, and problem constraints, convert constraints to sym_constraints, and add to self.sym_constraints if not present """
        sym_constraints = dict()
        
        k = 0
        for i in range(self.num_vars):
            varname = self.sym
        # Iterate through the constraints and apply the _sym_substitution function
        pass
    
    def _gen_code(self):
        code = "from gurobipy import *\n\n"
        # Define function name
        code += "def make_and_optimize_model():\n"
        # Add model
        code += """\tm = Model("Optimization_Problem")\n"""
        # Define Variables
        for key in self.sym_vars.keys():
            code+=f"""\t{key} = m.addVar(vtype = {self._get_var_type(key)}, name = '{key}')\n"""
        # Set Objective Function
        if self.goal == "maximize":
            goal_term = "GRB.MAXIMIZE"
        else:
            goal_term = "GRB.MINIMIZE"
        code +=f"""\tm.setObjective({self.objective_function_data['objective_function']}, {goal_term})\n"""
        # Add Constraints
        #for the constraints in sym_constraints
        for ctype, cdict in self.constraints.items():
            if ctype != 'int_types' and ctype !="total_binder":
                for cname, constraint in cdict.items():
                    code+=f"""\tm.addConstr({constraint['description']})\n"""
        # Optimize Model
        code += f"""\tm.optimize()\n"""
        code += "\treturn m"
        return code
    
    def _run_code(self):
        generated_fun = self.code
        # proceed to run the gurobi code
        local_scope = {}
        exec(generated_fun, globals())
        m = make_and_optimize_model()
        solution = dict()
        v_solution = dict()
        if m.status == GRB.OPTIMAL:
            for v in m.getVars():
                try:
                    v_solution.__setitem__(v.VarName, v.X)
                except AttributeError:
                    try:
                        v_solution.__setitem__(v.VarName, v.Xn)
                    except Exception as E:
                        print(f" Error {E} on input {v}")
                        v_solution.__setitem__(v.VarName,0)
            try:
                solution.__setitem__('Value',m.ObjVal)
            except AttributeError:
                try:
                    solution.__setitem__('Value',m.ObjNVal)
                except Exception as E:
                    print(f" Error {E} on input {m}")
        else:
            solution.__setitem__('Value','Infeasible')
        solution.__setitem__("Optimal solution",v_solution)
        return solution
    
    def _generate_sym_objective_function(self):
        """Given the current objective function statement, problem variables, and sym_vars, convert the objective function into """
        pass
    
    def _check_consistency(self) -> bool:
        """Method to check that the conversion methods losslessly convert between their strings and symbolic values """
        pass
    
    def load(self,address:str):
        pass
    
    def save(self,address:str):
        pass
    
    def update(self,external_update):
        if not isinstance(external_update, self.__class__):
            raise TypeError("Can only replace attributes with another instance of the same class")
        self.__dict__ = external_update.__dict__.copy()
    
