

# test_protocol_utils.py
# """
# Integration smoke test for protocol_utils.py (customized paths/leaf):
# - Generates a 5-problem tranche with deterministic seeding
# - Emits per-problem artifacts for model "gptoss"
# - Does NOT write sym/IR files (with_sym = False)
# - Uses custom root output path: ./utils_test_data_temp/
# - Uses custom leaf folder name: "utils_test_problems" (NOT the defaults)
# Run:
#     python test_protocol_utils.py
# """

# test_protocol_utils.py
"""
Integration smoke test for protocol_utils.py with custom paths/leaf:
- Generates a 5-problem tranche with deterministic seeding
- Emits per-problem artifacts for model "gptoss"
- Does NOT write sym/IR files (with_sym = False)
- Uses custom root output path: ./utils_test_data_temp/
- Uses custom leaf folder: utils_test_problems
Run:
    python ./src/test_protocol_utils.py
"""

import os
import json
from typing import Dict, Any

from protocol_utils import (
    TrancheConfig,
    make_tranche,
    save_model_outputs,
)


def _load_problems_from_tranche(tranche_dir: str):
    problems_dir = os.path.join(tranche_dir, "problems")
    ids = []
    dicts: Dict[str, Any] = {}
    for fname in sorted(os.listdir(problems_dir)):
        if fname.endswith(".json"):
            pid = fname[:-5]
            with open(os.path.join(problems_dir, fname), "r", encoding="utf-8") as f:
                pd = json.load(f)
            ids.append(pid)
            dicts[pid] = pd
    return ids, dicts


def _emit_code_from_problem_dict(pd: Dict[str, Any]) -> str:
    # Prefer Problem-generated code if present; else a minimal stub that compiles
    code = pd.get("gurobi_code")
    if isinstance(code, str) and code.strip():
        return code
    return """\
# Auto-generated test stub for utils_test_problems
def solve():
    return {"status": "unknown", "objective": None}
if __name__ == "__main__":
    print(solve())
"""


def main():
    # 1) Make a tiny tranche (N=5) under ./utils_test_data_temp/tranche_demo
    cfg = TrancheConfig(
        name="UTILS-TEST-LQ-mini",
        problem_kwargs=dict(
            # Reasonable defaults; adjust to your Problem API as needed
            deg_2_threshold=0.75,
            cts_threshold=0.5,
            mixed_int_threshold=0.75,
            max_threshold=0.5,
            int_coefs_only_threshold=0.5,
            triple_sign_1_threshold=1/3,
            triple_sign_2_threshold=2/3,
            pair_sign_threshold=1/2,
            include_exp=False,
            mheight=0,
            # Ensure presence even if the class doesn’t read it early
            resources_split_parameter=0.5,
        ),
    )

    tranche_root = os.path.join(os.getcwd(), "utils_test_data_temp", "tranche_demo")
    os.makedirs(tranche_root, exist_ok=True)

    card = make_tranche(cfg, seed=424242, n=5, out_dir=tranche_root, as_json=True)
    print("[OK] Minted tranche card:", json.dumps(card, indent=2))

    # 2) Load the 5 problems we just minted
    pids, pdicts = _load_problems_from_tranche(tranche_root)
    assert len(pids) == 5, f"Expected 5 problems, found {len(pids)}"
    print("[OK] Problem IDs:", pids)

    # 3) Build outputs for model 'gptoss' (no sym; with_sym=False)
    outputs = {}
    for pid in pids:
        pd = pdicts[pid]
        code = _emit_code_from_problem_dict(pd)
        outputs[pid] = {
            "code": code,
            "response_text": "auto-generated utils test artifact",
        }

    # Custom root & leaf: write under ./utils_test_data_temp/utils_test_problems/
    out_dir = save_model_outputs(
        dataset_tag="oproblems",        # tag is OK; leaf overrides below dictate the actual folder name
        with_sym=False,                 # per request
        model_name="gptoss",
        outputs=outputs,
        write_response_txt=True,
        ensure_dirs=True,
        # ---- custom pathing knobs ----
        output_save_path=os.path.join(os.getcwd(), "utils_test_data_temp"),
        base_leaf="utils_test_problems",     # NON-default leaf
        sym_leaf="utils_test_problems_sym",  # unused here
        altbase_leaf="utils_test_problems_nlp4lp",
        altsym_leaf="utils_test_problems_with_sym_nlp4lp",
        original_tag="oproblems",
        alt_tag="nlp4opt",
    )

    print("[OK] Wrote per-problem files to:", out_dir)

    # Sanity checks: *.py and *_response.txt
    expected_py = [os.path.join(out_dir, f"{pid}_gptoss.py") for pid in pids]
    missing_py = [p for p in expected_py if not os.path.exists(p)]
    assert not missing_py, f"Missing expected *.py files: {missing_py}"
    print("[OK] Verified expected *_gptoss.py files exist")

    expected_txt = [os.path.join(out_dir, f"{pid}_gptoss_response.txt") for pid in pids]
    missing_txt = [p for p in expected_txt if not os.path.exists(p)]
    assert not missing_txt, f"Missing expected response txt files: {missing_txt}"
    print("[OK] Verified expected *_gptoss_response.txt files exist")

    print("\nAll utils tests completed successfully.")
    print("Root dir:", os.path.join(os.getcwd(), "utils_test_data_temp"))
    print("Leaf dir:", out_dir)


if __name__ == "__main__":
    main()


# import os
# import json
# from typing import Dict, Any

# from protocol_utils import (
#     TrancheConfig,
#     make_tranche,
#     save_model_outputs,
# )

# # ---------------- Helpers ----------------

# def _load_problems_from_tranche(tranche_dir: str):
#     problems_dir = os.path.join(tranche_dir, "problems")
#     ids = []
#     dicts: Dict[str, Any] = {}
#     for fname in sorted(os.listdir(problems_dir)):
#         if fname.endswith(".json"):
#             pid = fname[:-5]
#             with open(os.path.join(problems_dir, fname), "r", encoding="utf-8") as f:
#                 pd = json.load(f)
#             ids.append(pid)
#             dicts[pid] = pd
#     return ids, dicts


# def _emit_code_from_problem_dict(pd: Dict[str, Any]) -> str:
#     """
#     Prefer Problem-generated solver code if present; otherwise emit a small stub so the file compiles.
#     """
#     code = pd.get("gurobi_code")
#     if isinstance(code, str) and code.strip():
#         return code

#     # Fallback: minimal stub
#     return """\
# # Auto-generated test stub for utils_test_problems
# def solve():
#     return {"status": "unknown", "objective": None}
# if __name__ == "__main__":
#     print(solve())
# """


# def main():
#     # ----- 1) Make a tiny tranche (N=5) under ./utils_test_data_temp/tranche_demo -----
#     cfg = TrancheConfig(
#         name="UTILS-TEST-LQ-mini",
#         problem_kwargs=dict(
#             # Use realistic defaults from your Problem; adjust if needed.
#             deg_2_threshold=0.75,
#             cts_threshold=0.5,
#             mixed_int_threshold=0.75,
#             max_threshold=0.5,
#             int_coefs_only_threshold=0.5,
#             triple_sign_1_threshold=1/3,
#             triple_sign_2_threshold=2/3,
#             pair_sign_threshold=1/2,
#             include_exp=False,
#             mheight=0,
#             resources_split_parameter=0.5,
#         ),
#     )

#     tranche_root = os.path.join(os.getcwd(), "utils_test_data_temp", "tranche_demo")
#     os.makedirs(tranche_root, exist_ok=True)

#     card = make_tranche(cfg, seed=424242, n=5, out_dir=tranche_root, as_json=True)
#     print("[OK] Minted tranche card:", json.dumps(card, indent=2))

#     # ----- 2) Load the 5 problems we just minted -----
#     pids, pdicts = _load_problems_from_tranche(tranche_root)
#     assert len(pids) == 5, f"Expected 5 problems, found {len(pids)}"
#     print("[OK] Problem IDs:", pids)

#     # ----- 3) Build outputs for model 'gptoss' (no sym; with_sym=False) -----
#     outputs = {}
#     for pid in pids:
#         pd = pdicts[pid]
#         code = _emit_code_from_problem_dict(pd)
#         outputs[pid] = {
#             "code": code,
#             # No "sym" key on purpose (with_sym=False)
#             "response_text": "auto-generated utils test artifact",
#         }

#     # Custom root & leaf: write under ./utils_test_data_temp/utils_test_problems/
#     out_dir = save_model_outputs(
#         dataset_tag="oproblems",        # tag is ignored for custom leaf override below
#         with_sym=False,                 # <- REQUIRED by your request
#         model_name="gptoss",            # <- model under test
#         outputs=outputs,
#         write_response_txt=True,
#         ensure_dirs=True,
#         # --- Custom pathing knobs ---
#         output_save_path=os.path.join(os.getcwd(), "utils_test_data_temp"),
#         base_leaf="utils_test_problems",     # <- NOT the default "baseline_0shot"
#         sym_leaf="utils_test_problems_sym",  # unused here (with_sym=False)
#         altbase_leaf="utils_test_problems_nlp4lp",   # unused here
#         altsym_leaf="utils_test_problems_with_sym_nlp4lp",  # unused here
#         original_tag="oproblems",
#         alt_tag="nlp4opt",
#     )

#     print("[OK] Wrote per-problem files to:", out_dir)

#     # Sanity: verify the expected filenames exist
#     expected = [os.path.join(out_dir, f"{pid}_gptoss.py") for pid in pids]
#     missing = [p for p in expected if not os.path.exists(p)]
#     assert not missing, f"Missing expected files: {missing}"
#     print("[OK] Verified expected *_gptoss.py files exist")

#     # Optionally verify response text files
#     expected_txt = [os.path.join(out_dir, f"{pid}_gptoss_response.txt") for pid in pids]
#     missing_txt = [p for p in expected_txt if not os.path.exists(p)]
#     assert not missing_txt, f"Missing expected response txt files: {missing_txt}"
#     print("[OK] Verified expected *_gptoss_response.txt files exist")

#     print("\nAll utils tests completed successfully.")
#     print("Root dir:", os.path.join(os.getcwd(), "utils_test_data_temp"))
#     print("Leaf dir:", out_dir)


# if __name__ == "__main__":
#     main()
