
#ifndef _testratinterpolation_h
#define _testratinterpolation_h

#include <stdio.h>
#include "ap.h"
#include "amp.h"
#include "tsort.h"
#include "ratinterpolation.h"
#include "blas.h"
#include "reflections.h"
#include "creflections.h"
#include "hqrnd.h"
#include "matgen.h"
#include "ablasf.h"
#include "ablas.h"
#include "trfac.h"
#include "trlinsolve.h"
#include "safesolve.h"
#include "rcond.h"
#include "matinv.h"
#include "hblas.h"
#include "sblas.h"
#include "ortfac.h"
#include "rotations.h"
#include "bdsvd.h"
#include "svd.h"
#include "xblas.h"
#include "densesolver.h"
#include "linmin.h"
#include "minlbfgs.h"
#include "minlm.h"
#include "lsfit.h"
#include "ratint.h"
namespace testratinterpolation
{
    template<unsigned int Precision>
    bool testri(bool silent);
    template<unsigned int Precision>
    void poldiff2(const ap::template_1d_array< amp::ampf<Precision> >& x,
        ap::template_1d_array< amp::ampf<Precision> > f,
        int n,
        amp::ampf<Precision> t,
        amp::ampf<Precision>& p,
        amp::ampf<Precision>& dp,
        amp::ampf<Precision>& d2p);
    template<unsigned int Precision>
    void brcunset(ratint::barycentricinterpolant<Precision>& b);
    template<unsigned int Precision>
    bool is1dsolution(int n,
        const ap::template_1d_array< amp::ampf<Precision> >& y,
        const ap::template_1d_array< amp::ampf<Precision> >& w,
        amp::ampf<Precision> c);
    template<unsigned int Precision>
    bool testratinterpolation_test_silent();
    template<unsigned int Precision>
    bool testratinterpolation_test();


    template<unsigned int Precision>
    bool testri(bool silent)
    {
        bool result;
        bool waserrors;
        bool bcerrors;
        bool nperrors;
        bool fiterrors;
        amp::ampf<Precision> threshold;
        amp::ampf<Precision> lipschitztol;
        int maxn;
        int passcount;
        ratint::barycentricinterpolant<Precision> b1;
        ratint::barycentricinterpolant<Precision> b2;
        ap::template_1d_array< amp::ampf<Precision> > x;
        ap::template_1d_array< amp::ampf<Precision> > x2;
        ap::template_1d_array< amp::ampf<Precision> > y;
        ap::template_1d_array< amp::ampf<Precision> > y2;
        ap::template_1d_array< amp::ampf<Precision> > w;
        ap::template_1d_array< amp::ampf<Precision> > w2;
        ap::template_1d_array< amp::ampf<Precision> > xc;
        ap::template_1d_array< amp::ampf<Precision> > yc;
        ap::template_1d_array< int > dc;
        amp::ampf<Precision> h;
        amp::ampf<Precision> s1;
        amp::ampf<Precision> s2;
        bool bsame;
        int n;
        int m;
        int n2;
        int i;
        int j;
        int k;
        int d;
        int pass;
        amp::ampf<Precision> err;
        amp::ampf<Precision> maxerr;
        amp::ampf<Precision> t;
        amp::ampf<Precision> a;
        amp::ampf<Precision> b;
        amp::ampf<Precision> s;
        amp::ampf<Precision> v;
        amp::ampf<Precision> v0;
        amp::ampf<Precision> v1;
        amp::ampf<Precision> v2;
        amp::ampf<Precision> v3;
        amp::ampf<Precision> d0;
        amp::ampf<Precision> d1;
        amp::ampf<Precision> d2;
        int info;
        int info2;
        amp::ampf<Precision> xmin;
        amp::ampf<Precision> xmax;
        amp::ampf<Precision> refrms;
        amp::ampf<Precision> refavg;
        amp::ampf<Precision> refavgrel;
        amp::ampf<Precision> refmax;
        ap::template_1d_array< amp::ampf<Precision> > ra;
        ap::template_1d_array< amp::ampf<Precision> > ra2;
        int ralen;
        ratint::barycentricfitreport<Precision> rep;
        ratint::barycentricfitreport<Precision> rep2;
        ratint::barycentricinterpolant<Precision> b3;
        ratint::barycentricinterpolant<Precision> b4;


        nperrors = false;
        bcerrors = false;
        fiterrors = false;
        waserrors = false;
        
        //
        // PassCount        number of repeated passes
        // Threshold        error tolerance
        // LipschitzTol     Lipschitz constant increase allowed
        //                  when calculating constant on a twice denser grid
        //
        passcount = 5;
        maxn = 15;
        threshold = 1000000*amp::ampf<Precision>::getAlgoPascalEpsilon();
        lipschitztol = amp::ampf<Precision>("1.3");
        
        //
        // Basic barycentric functions
        //
        for(n=1; n<=10; n++)
        {
            
            //
            // randomized tests
            //
            for(pass=1; pass<=passcount; pass++)
            {
                
                //
                // generate weights from polynomial interpolation
                //
                v0 = 1+amp::ampf<Precision>("0.4")*amp::ampf<Precision>::getRandom()-amp::ampf<Precision>("0.2");
                v1 = 2*amp::ampf<Precision>::getRandom()-1;
                v2 = 2*amp::ampf<Precision>::getRandom()-1;
                v3 = 2*amp::ampf<Precision>::getRandom()-1;
                x.setlength(n);
                y.setlength(n);
                w.setlength(n);
                for(i=0; i<=n-1; i++)
                {
                    if( n==1 )
                    {
                        x(i) = 0;
                    }
                    else
                    {
                        x(i) = v0*amp::cos<Precision>(i*amp::pi<Precision>()/(n-1));
                    }
                    y(i) = amp::sin<Precision>(v1*x(i))+amp::cos<Precision>(v2*x(i))+amp::exp<Precision>(v3*x(i));
                }
                for(j=0; j<=n-1; j++)
                {
                    w(j) = 1;
                    for(k=0; k<=n-1; k++)
                    {
                        if( k!=j )
                        {
                            w(j) = w(j)/(x(j)-x(k));
                        }
                    }
                }
                ratint::barycentricbuildxyw<Precision>(x, y, w, n, b1);
                
                //
                // unpack, then pack again and compare
                //
                brcunset<Precision>(b2);
                ratint::barycentricunpack<Precision>(b1, n2, x2, y2, w2);
                bcerrors = bcerrors || n2!=n;
                ratint::barycentricbuildxyw<Precision>(x2, y2, w2, n2, b2);
                t = 2*amp::ampf<Precision>::getRandom()-1;
                bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, t)-ratint::barycentriccalc<Precision>(b2, t))>threshold;
                
                //
                // serialize, unserialize, compare
                //
                brcunset<Precision>(b2);
                ratint::barycentricserialize<Precision>(b1, ra, ralen);
                ra2.setlength(ralen);
                amp::vmove(ra2.getvector(0, ralen-1), ra.getvector(0, ralen-1));
                ratint::barycentricunserialize<Precision>(ra2, b2);
                t = 2*amp::ampf<Precision>::getRandom()-1;
                bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, t)-ratint::barycentriccalc<Precision>(b2, t))>threshold;
                
                //
                // copy, compare
                //
                brcunset<Precision>(b2);
                ratint::barycentriccopy<Precision>(b1, b2);
                t = 2*amp::ampf<Precision>::getRandom()-1;
                bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, t)-ratint::barycentriccalc<Precision>(b2, t))>threshold;
                
                //
                // test interpolation properties
                //
                for(i=0; i<=n-1; i++)
                {
                    
                    //
                    // test interpolation at nodes
                    //
                    bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, x(i))-y(i))>threshold*amp::abs<Precision>(y(i));
                    
                    //
                    // compare with polynomial interpolation
                    //
                    t = 2*amp::ampf<Precision>::getRandom()-1;
                    poldiff2<Precision>(x, y, n, t, v0, v1, v2);
                    bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, t)-v0)>threshold*amp::maximum<Precision>(amp::abs<Precision>(v0), amp::ampf<Precision>(1));
                    
                    //
                    // test continuity between nodes
                    // calculate Lipschitz constant on two grids -
                    // dense and even more dense. If Lipschitz constant
                    // on a denser grid is significantly increased,
                    // continuity test is failed
                    //
                    t = amp::ampf<Precision>("3.0");
                    k = 100;
                    s1 = 0;
                    for(j=0; j<=k-1; j++)
                    {
                        v1 = x(i)+(t-x(i))*j/k;
                        v2 = x(i)+(t-x(i))*(j+1)/k;
                        s1 = amp::maximum<Precision>(s1, amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, v2)-ratint::barycentriccalc<Precision>(b1, v1))/amp::abs<Precision>(v2-v1));
                    }
                    k = 2*k;
                    s2 = 0;
                    for(j=0; j<=k-1; j++)
                    {
                        v1 = x(i)+(t-x(i))*j/k;
                        v2 = x(i)+(t-x(i))*(j+1)/k;
                        s2 = amp::maximum<Precision>(s2, amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, v2)-ratint::barycentriccalc<Precision>(b1, v1))/amp::abs<Precision>(v2-v1));
                    }
                    bcerrors = bcerrors || s2>lipschitztol*s1 && s1>threshold*k;
                }
                
                //
                // test differentiation properties
                //
                for(i=0; i<=n-1; i++)
                {
                    t = 2*amp::ampf<Precision>::getRandom()-1;
                    poldiff2<Precision>(x, y, n, t, v0, v1, v2);
                    d0 = 0;
                    d1 = 0;
                    d2 = 0;
                    ratint::barycentricdiff1<Precision>(b1, t, d0, d1);
                    bcerrors = bcerrors || amp::abs<Precision>(v0-d0)>threshold*amp::maximum<Precision>(amp::abs<Precision>(v0), amp::ampf<Precision>(1));
                    bcerrors = bcerrors || amp::abs<Precision>(v1-d1)>threshold*amp::maximum<Precision>(amp::abs<Precision>(v1), amp::ampf<Precision>(1));
                    d0 = 0;
                    d1 = 0;
                    d2 = 0;
                    ratint::barycentricdiff2<Precision>(b1, t, d0, d1, d2);
                    bcerrors = bcerrors || amp::abs<Precision>(v0-d0)>threshold*amp::maximum<Precision>(amp::abs<Precision>(v0), amp::ampf<Precision>(1));
                    bcerrors = bcerrors || amp::abs<Precision>(v1-d1)>threshold*amp::maximum<Precision>(amp::abs<Precision>(v1), amp::ampf<Precision>(1));
                    bcerrors = bcerrors || amp::abs<Precision>(v2-d2)>amp::sqrt<Precision>(threshold)*amp::maximum<Precision>(amp::abs<Precision>(v2), amp::ampf<Precision>(1));
                }
                
                //
                // test linear translation
                //
                t = 2*amp::ampf<Precision>::getRandom()-1;
                a = 2*amp::ampf<Precision>::getRandom()-1;
                b = 2*amp::ampf<Precision>::getRandom()-1;
                brcunset<Precision>(b2);
                ratint::barycentriccopy<Precision>(b1, b2);
                ratint::barycentriclintransx<Precision>(b2, a, b);
                bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, a*t+b)-ratint::barycentriccalc<Precision>(b2, t))>threshold;
                a = 0;
                b = 2*amp::ampf<Precision>::getRandom()-1;
                brcunset<Precision>(b2);
                ratint::barycentriccopy<Precision>(b1, b2);
                ratint::barycentriclintransx<Precision>(b2, a, b);
                bcerrors = bcerrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, a*t+b)-ratint::barycentriccalc<Precision>(b2, t))>threshold;
                a = 2*amp::ampf<Precision>::getRandom()-1;
                b = 2*amp::ampf<Precision>::getRandom()-1;
                brcunset<Precision>(b2);
                ratint::barycentriccopy<Precision>(b1, b2);
                ratint::barycentriclintransy<Precision>(b2, a, b);
                bcerrors = bcerrors || amp::abs<Precision>(a*ratint::barycentriccalc<Precision>(b1, t)+b-ratint::barycentriccalc<Precision>(b2, t))>threshold;
            }
        }
        for(pass=0; pass<=3; pass++)
        {
            
            //
            // Crash-test: small numbers, large numbers
            //
            x.setlength(4);
            y.setlength(4);
            w.setlength(4);
            h = 1;
            if( pass%2==0 )
            {
                h = 100*amp::ampf<Precision>::getAlgoPascalMinNumber();
            }
            if( pass%2==1 )
            {
                h = amp::ampf<Precision>("0.01")*amp::ampf<Precision>::getAlgoPascalMaxNumber();
            }
            x(0) = 0*h;
            x(1) = 1*h;
            x(2) = 2*h;
            x(3) = 3*h;
            y(0) = 0*h;
            y(1) = 1*h;
            y(2) = 2*h;
            y(3) = 3*h;
            w(0) = -1/(x(1)-x(0));
            w(1) = +1*(1/(x(1)-x(0))+1/(x(2)-x(1)));
            w(2) = -1*(1/(x(2)-x(1))+1/(x(3)-x(2)));
            w(3) = +1/(x(3)-x(2));
            if( pass/2==0 )
            {
                v0 = 0;
            }
            if( pass/2==1 )
            {
                v0 = amp::ampf<Precision>("0.6")*h;
            }
            ratint::barycentricbuildxyw<Precision>(x, y, w, 4, b1);
            t = ratint::barycentriccalc<Precision>(b1, v0);
            d0 = 0;
            d1 = 0;
            d2 = 0;
            ratint::barycentricdiff1<Precision>(b1, v0, d0, d1);
            bcerrors = bcerrors || amp::abs<Precision>(t-v0)>threshold*v0;
            bcerrors = bcerrors || amp::abs<Precision>(d0-v0)>threshold*v0;
            bcerrors = bcerrors || amp::abs<Precision>(d1-1)>1000*threshold;
        }
        
        //
        // crash test: large abscissas, small argument
        //
        // test for errors in D0 is not very strict
        // because renormalization used in Diff1()
        // destroys part of precision.
        //
        x.setlength(4);
        y.setlength(4);
        w.setlength(4);
        h = amp::ampf<Precision>("0.01")*amp::ampf<Precision>::getAlgoPascalMaxNumber();
        x(0) = 0*h;
        x(1) = 1*h;
        x(2) = 2*h;
        x(3) = 3*h;
        y(0) = 0*h;
        y(1) = 1*h;
        y(2) = 2*h;
        y(3) = 3*h;
        w(0) = -1/(x(1)-x(0));
        w(1) = +1*(1/(x(1)-x(0))+1/(x(2)-x(1)));
        w(2) = -1*(1/(x(2)-x(1))+1/(x(3)-x(2)));
        w(3) = +1/(x(3)-x(2));
        v0 = 100*amp::ampf<Precision>::getAlgoPascalMinNumber();
        ratint::barycentricbuildxyw<Precision>(x, y, w, 4, b1);
        t = ratint::barycentriccalc<Precision>(b1, v0);
        d0 = 0;
        d1 = 0;
        d2 = 0;
        ratint::barycentricdiff1<Precision>(b1, v0, d0, d1);
        bcerrors = bcerrors || amp::abs<Precision>(t)>v0*(1+threshold);
        bcerrors = bcerrors || amp::abs<Precision>(d0)>v0*(1+threshold);
        bcerrors = bcerrors || amp::abs<Precision>(d1-1)>1000*threshold;
        
        //
        // crash test: test safe barycentric formula
        //
        x.setlength(4);
        y.setlength(4);
        w.setlength(4);
        h = 2*amp::ampf<Precision>::getAlgoPascalMinNumber();
        x(0) = 0*h;
        x(1) = 1*h;
        x(2) = 2*h;
        x(3) = 3*h;
        y(0) = 0*h;
        y(1) = 1*h;
        y(2) = 2*h;
        y(3) = 3*h;
        w(0) = -1/(x(1)-x(0));
        w(1) = +1*(1/(x(1)-x(0))+1/(x(2)-x(1)));
        w(2) = -1*(1/(x(2)-x(1))+1/(x(3)-x(2)));
        w(3) = +1/(x(3)-x(2));
        v0 = amp::ampf<Precision>::getAlgoPascalMinNumber();
        ratint::barycentricbuildxyw<Precision>(x, y, w, 4, b1);
        t = ratint::barycentriccalc<Precision>(b1, v0);
        bcerrors = bcerrors || amp::abs<Precision>(t-v0)/v0>threshold;
        
        //
        // Testing "No Poles" interpolation
        //
        maxerr = 0;
        for(pass=1; pass<=passcount-1; pass++)
        {
            x.setlength(1);
            y.setlength(1);
            x(0) = 2*amp::ampf<Precision>::getRandom()-1;
            y(0) = 2*amp::ampf<Precision>::getRandom()-1;
            ratint::barycentricbuildfloaterhormann<Precision>(x, y, 1, 1, b1);
            maxerr = amp::maximum<Precision>(maxerr, amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, 2*amp::ampf<Precision>::getRandom()-1)-y(0)));
        }
        for(n=2; n<=10; n++)
        {
            
            //
            // compare interpolant built by subroutine
            // with interpolant built by hands
            //
            x.setlength(n);
            y.setlength(n);
            w.setlength(n);
            w2.setlength(n);
            
            //
            // D=1, non-equidistant nodes
            //
            for(pass=1; pass<=passcount; pass++)
            {
                
                //
                // Initialize X, Y, W
                //
                a = -1-1*amp::ampf<Precision>::getRandom();
                b = +1+1*amp::ampf<Precision>::getRandom();
                for(i=0; i<=n-1; i++)
                {
                    x(i) = amp::atan<Precision>((b-a)*i/(n-1)+a);
                }
                for(i=0; i<=n-1; i++)
                {
                    y(i) = 2*amp::ampf<Precision>::getRandom()-1;
                }
                w(0) = -1/(x(1)-x(0));
                s = 1;
                for(i=1; i<=n-2; i++)
                {
                    w(i) = s*(1/(x(i)-x(i-1))+1/(x(i+1)-x(i)));
                    s = -s;
                }
                w(n-1) = s/(x(n-1)-x(n-2));
                for(i=0; i<=n-1; i++)
                {
                    k = ap::randominteger(n);
                    if( k!=i )
                    {
                        t = x(i);
                        x(i) = x(k);
                        x(k) = t;
                        t = y(i);
                        y(i) = y(k);
                        y(k) = t;
                        t = w(i);
                        w(i) = w(k);
                        w(k) = t;
                    }
                }
                
                //
                // Build and test
                //
                ratint::barycentricbuildfloaterhormann<Precision>(x, y, n, 1, b1);
                ratint::barycentricbuildxyw<Precision>(x, y, w, n, b2);
                for(i=1; i<=2*n; i++)
                {
                    t = a+(b-a)*amp::ampf<Precision>::getRandom();
                    maxerr = amp::maximum<Precision>(maxerr, amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, t)-ratint::barycentriccalc<Precision>(b2, t)));
                }
            }
            
            //
            // D = 0, 1, 2. Equidistant nodes.
            //
            for(d=0; d<=2; d++)
            {
                for(pass=1; pass<=passcount; pass++)
                {
                    
                    //
                    // Skip incorrect (N,D) pairs
                    //
                    if( n<2*d )
                    {
                        continue;
                    }
                    
                    //
                    // Initialize X, Y, W
                    //
                    a = -1-1*amp::ampf<Precision>::getRandom();
                    b = +1+1*amp::ampf<Precision>::getRandom();
                    for(i=0; i<=n-1; i++)
                    {
                        x(i) = (b-a)*i/(n-1)+a;
                    }
                    for(i=0; i<=n-1; i++)
                    {
                        y(i) = 2*amp::ampf<Precision>::getRandom()-1;
                    }
                    s = 1;
                    if( d==0 )
                    {
                        for(i=0; i<=n-1; i++)
                        {
                            w(i) = s;
                            s = -s;
                        }
                    }
                    if( d==1 )
                    {
                        w(0) = -s;
                        for(i=1; i<=n-2; i++)
                        {
                            w(i) = 2*s;
                            s = -s;
                        }
                        w(n-1) = s;
                    }
                    if( d==2 )
                    {
                        w(0) = s;
                        w(1) = -3*s;
                        for(i=2; i<=n-3; i++)
                        {
                            w(i) = 4*s;
                            s = -s;
                        }
                        w(n-2) = 3*s;
                        w(n-1) = -s;
                    }
                    
                    //
                    // Mix
                    //
                    for(i=0; i<=n-1; i++)
                    {
                        k = ap::randominteger(n);
                        if( k!=i )
                        {
                            t = x(i);
                            x(i) = x(k);
                            x(k) = t;
                            t = y(i);
                            y(i) = y(k);
                            y(k) = t;
                            t = w(i);
                            w(i) = w(k);
                            w(k) = t;
                        }
                    }
                    
                    //
                    // Build and test
                    //
                    ratint::barycentricbuildfloaterhormann<Precision>(x, y, n, d, b1);
                    ratint::barycentricbuildxyw<Precision>(x, y, w, n, b2);
                    for(i=1; i<=2*n; i++)
                    {
                        t = a+(b-a)*amp::ampf<Precision>::getRandom();
                        maxerr = amp::maximum<Precision>(maxerr, amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, t)-ratint::barycentriccalc<Precision>(b2, t)));
                    }
                }
            }
        }
        if( maxerr>threshold )
        {
            nperrors = true;
        }
        
        //
        // Test rational fitting:
        //
        for(pass=1; pass<=passcount; pass++)
        {
            for(n=2; n<=maxn; n++)
            {
                
                //
                // N=M+K fitting (i.e. interpolation)
                //
                for(k=0; k<=n-1; k++)
                {
                    x.setlength(n-k);
                    y.setlength(n-k);
                    w.setlength(n-k);
                    if( k>0 )
                    {
                        xc.setlength(k);
                        yc.setlength(k);
                        dc.setlength(k);
                    }
                    for(i=0; i<=n-k-1; i++)
                    {
                        x(i) = amp::ampf<Precision>(i)/(amp::ampf<Precision>(n-1));
                        y(i) = 2*amp::ampf<Precision>::getRandom()-1;
                        w(i) = 1+amp::ampf<Precision>::getRandom();
                    }
                    for(i=0; i<=k-1; i++)
                    {
                        xc(i) = (amp::ampf<Precision>(n-k+i))/(amp::ampf<Precision>(n-1));
                        yc(i) = 2*amp::ampf<Precision>::getRandom()-1;
                        dc(i) = 0;
                    }
                    ratint::barycentricfitfloaterhormannwc<Precision>(x, y, w, n-k, xc, yc, dc, k, n, info, b1, rep);
                    if( info<=0 )
                    {
                        fiterrors = true;
                    }
                    else
                    {
                        for(i=0; i<=n-k-1; i++)
                        {
                            fiterrors = fiterrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, x(i))-y(i))>threshold;
                        }
                        for(i=0; i<=k-1; i++)
                        {
                            fiterrors = fiterrors || amp::abs<Precision>(ratint::barycentriccalc<Precision>(b1, xc(i))-yc(i))>threshold;
                        }
                    }
                }
                
                //
                // Testing constraints on derivatives:
                // * several M's are tried
                // * several K's are tried - 1, 2.
                // * constraints at the ends of the interval
                //
                for(m=3; m<=5; m++)
                {
                    for(k=1; k<=2; k++)
                    {
                        x.setlength(n);
                        y.setlength(n);
                        w.setlength(n);
                        xc.setlength(2);
                        yc.setlength(2);
                        dc.setlength(2);
                        for(i=0; i<=n-1; i++)
                        {
                            x(i) = 2*amp::ampf<Precision>::getRandom()-1;
                            y(i) = 2*amp::ampf<Precision>::getRandom()-1;
                            w(i) = 1+amp::ampf<Precision>::getRandom();
                        }
                        xc(0) = -1;
                        yc(0) = 2*amp::ampf<Precision>::getRandom()-1;
                        dc(0) = 0;
                        xc(1) = +1;
                        yc(1) = 2*amp::ampf<Precision>::getRandom()-1;
                        dc(1) = 0;
                        ratint::barycentricfitfloaterhormannwc<Precision>(x, y, w, n, xc, yc, dc, k, m, info, b1, rep);
                        if( info<=0 )
                        {
                            fiterrors = true;
                        }
                        else
                        {
                            for(i=0; i<=k-1; i++)
                            {
                                ratint::barycentricdiff1<Precision>(b1, xc(i), v0, v1);
                                fiterrors = fiterrors || amp::abs<Precision>(v0-yc(i))>threshold;
                            }
                        }
                    }
                }
            }
        }
        for(m=2; m<=8; m++)
        {
            for(pass=1; pass<=passcount; pass++)
            {
                
                //
                // General fitting
                //
                // interpolating function through M nodes should have
                // greater RMS error than fitting it through the same M nodes
                //
                n = 100;
                x2.setlength(n);
                y2.setlength(n);
                w2.setlength(n);
                xmin = amp::ampf<Precision>::getAlgoPascalMaxNumber();
                xmax = -amp::ampf<Precision>::getAlgoPascalMaxNumber();
                for(i=0; i<=n-1; i++)
                {
                    x2(i) = 2*amp::pi<Precision>()*amp::ampf<Precision>::getRandom();
                    y2(i) = amp::sin<Precision>(x2(i));
                    w2(i) = 1;
                    xmin = amp::minimum<Precision>(xmin, x2(i));
                    xmax = amp::maximum<Precision>(xmax, x2(i));
                }
                x.setlength(m);
                y.setlength(m);
                for(i=0; i<=m-1; i++)
                {
                    x(i) = xmin+(xmax-xmin)*i/(m-1);
                    y(i) = amp::sin<Precision>(x(i));
                }
                ratint::barycentricbuildfloaterhormann<Precision>(x, y, m, 3, b1);
                ratint::barycentricfitfloaterhormannwc<Precision>(x2, y2, w2, n, xc, yc, dc, 0, m, info, b2, rep);
                if( info<=0 )
                {
                    fiterrors = true;
                }
                else
                {
                    
                    //
                    // calculate B1 (interpolant) RMS error, compare with B2 error
                    //
                    v1 = 0;
                    v2 = 0;
                    for(i=0; i<=n-1; i++)
                    {
                        v1 = v1+amp::sqr<Precision>(ratint::barycentriccalc<Precision>(b1, x2(i))-y2(i));
                        v2 = v2+amp::sqr<Precision>(ratint::barycentriccalc<Precision>(b2, x2(i))-y2(i));
                    }
                    v1 = amp::sqrt<Precision>(v1/n);
                    v2 = amp::sqrt<Precision>(v2/n);
                    fiterrors = fiterrors || v2>v1;
                    fiterrors = fiterrors || amp::abs<Precision>(v2-rep.rmserror)>threshold;
                }
                
                //
                // compare weighted and non-weighted
                //
                n = 20;
                x.setlength(n);
                y.setlength(n);
                w.setlength(n);
                for(i=0; i<=n-1; i++)
                {
                    x(i) = 2*amp::ampf<Precision>::getRandom()-1;
                    y(i) = 2*amp::ampf<Precision>::getRandom()-1;
                    w(i) = 1;
                }
                ratint::barycentricfitfloaterhormannwc<Precision>(x, y, w, n, xc, yc, dc, 0, m, info, b1, rep);
                ratint::barycentricfitfloaterhormann<Precision>(x, y, n, m, info2, b2, rep2);
                if( info<=0 || info2<=0 )
                {
                    fiterrors = true;
                }
                else
                {
                    
                    //
                    // calculate B1 (interpolant), compare with B2
                    // compare RMS errors
                    //
                    t = 2*amp::ampf<Precision>::getRandom()-1;
                    v1 = ratint::barycentriccalc<Precision>(b1, t);
                    v2 = ratint::barycentriccalc<Precision>(b2, t);
                    fiterrors = fiterrors || v2!=v1;
                    fiterrors = fiterrors || rep.rmserror!=rep2.rmserror;
                    fiterrors = fiterrors || rep.avgerror!=rep2.avgerror;
                    fiterrors = fiterrors || rep.avgrelerror!=rep2.avgrelerror;
                    fiterrors = fiterrors || rep.maxerror!=rep2.maxerror;
                }
            }
        }
        for(pass=1; pass<=passcount; pass++)
        {
            ap::ap_error::make_assertion(passcount>=2);
            
            //
            // solve simple task (all X[] are the same, Y[] are specially
            // calculated to ensure simple form of all types of errors)
            // and check correctness of the errors calculated by subroutines
            //
            // First pass is done with zero Y[], other passes - with random Y[].
            // It should test both ability to correctly calculate errors and
            // ability to not fail while working with zeros :)
            //
            n = 4;
            if( pass==1 )
            {
                v1 = 0;
                v2 = 0;
                v = 0;
            }
            else
            {
                v1 = amp::ampf<Precision>::getRandom();
                v2 = amp::ampf<Precision>::getRandom();
                v = 1+amp::ampf<Precision>::getRandom();
            }
            x.setlength(4);
            y.setlength(4);
            w.setlength(4);
            x(0) = 0;
            y(0) = v-v2;
            w(0) = 1;
            x(1) = 0;
            y(1) = v-v1;
            w(1) = 1;
            x(2) = 0;
            y(2) = v+v1;
            w(2) = 1;
            x(3) = 0;
            y(3) = v+v2;
            w(3) = 1;
            refrms = amp::sqrt<Precision>((amp::sqr<Precision>(v1)+amp::sqr<Precision>(v2))/2);
            refavg = (amp::abs<Precision>(v1)+amp::abs<Precision>(v2))/2;
            if( pass==1 )
            {
                refavgrel = 0;
            }
            else
            {
                refavgrel = amp::ampf<Precision>("0.25")*(amp::abs<Precision>(v2)/amp::abs<Precision>(v-v2)+amp::abs<Precision>(v1)/amp::abs<Precision>(v-v1)+amp::abs<Precision>(v1)/amp::abs<Precision>(v+v1)+amp::abs<Precision>(v2)/amp::abs<Precision>(v+v2));
            }
            refmax = amp::maximum<Precision>(v1, v2);
            
            //
            // Test errors correctness
            //
            ratint::barycentricfitfloaterhormann<Precision>(x, y, 4, 2, info, b1, rep);
            if( info<=0 )
            {
                fiterrors = true;
            }
            else
            {
                s = ratint::barycentriccalc<Precision>(b1, amp::ampf<Precision>(0));
                fiterrors = fiterrors || amp::abs<Precision>(s-v)>threshold;
                fiterrors = fiterrors || amp::abs<Precision>(rep.rmserror-refrms)>threshold;
                fiterrors = fiterrors || amp::abs<Precision>(rep.avgerror-refavg)>threshold;
                fiterrors = fiterrors || amp::abs<Precision>(rep.avgrelerror-refavgrel)>threshold;
                fiterrors = fiterrors || amp::abs<Precision>(rep.maxerror-refmax)>threshold;
            }
        }
        
        //
        // report
        //
        waserrors = bcerrors || nperrors || fiterrors;
        if( !silent )
        {
            printf("TESTING RATIONAL INTERPOLATION\n");
            printf("BASIC BARYCENTRIC FUNCTIONS:             ");
            if( bcerrors )
            {
                printf("FAILED\n");
            }
            else
            {
                printf("OK\n");
            }
            printf("FLOATER-HORMANN:                         ");
            if( nperrors )
            {
                printf("FAILED\n");
            }
            else
            {
                printf("OK\n");
            }
            printf("RATIONAL FITTING:                        ");
            if( fiterrors )
            {
                printf("FAILED\n");
            }
            else
            {
                printf("OK\n");
            }
            if( waserrors )
            {
                printf("TEST FAILED\n");
            }
            else
            {
                printf("TEST PASSED\n");
            }
            printf("\n\n");
        }
        
        //
        // end
        //
        result = !waserrors;
        return result;
    }


    template<unsigned int Precision>
    void poldiff2(const ap::template_1d_array< amp::ampf<Precision> >& x,
        ap::template_1d_array< amp::ampf<Precision> > f,
        int n,
        amp::ampf<Precision> t,
        amp::ampf<Precision>& p,
        amp::ampf<Precision>& dp,
        amp::ampf<Precision>& d2p)
    {
        int m;
        int i;
        ap::template_1d_array< amp::ampf<Precision> > df;
        ap::template_1d_array< amp::ampf<Precision> > d2f;


        n = n-1;
        df.setbounds(0, n);
        d2f.setbounds(0, n);
        for(i=0; i<=n; i++)
        {
            d2f(i) = 0;
            df(i) = 0;
        }
        for(m=1; m<=n; m++)
        {
            for(i=0; i<=n-m; i++)
            {
                d2f(i) = ((t-x(i+m))*d2f(i)+(x(i)-t)*d2f(i+1)+2*df(i)-2*df(i+1))/(x(i)-x(i+m));
                df(i) = ((t-x(i+m))*df(i)+f(i)+(x(i)-t)*df(i+1)-f(i+1))/(x(i)-x(i+m));
                f(i) = ((t-x(i+m))*f(i)+(x(i)-t)*f(i+1))/(x(i)-x(i+m));
            }
        }
        p = f(0);
        dp = df(0);
        d2p = d2f(0);
    }


    template<unsigned int Precision>
    void brcunset(ratint::barycentricinterpolant<Precision>& b)
    {
        ap::template_1d_array< amp::ampf<Precision> > x;
        ap::template_1d_array< amp::ampf<Precision> > y;
        ap::template_1d_array< amp::ampf<Precision> > w;


        x.setlength(1);
        y.setlength(1);
        w.setlength(1);
        x(0) = 0;
        y(0) = 0;
        w(0) = 1;
        ratint::barycentricbuildxyw<Precision>(x, y, w, 1, b);
    }


    /*************************************************************************
    Tests whether constant C is solution of 1D LLS problem
    *************************************************************************/
    template<unsigned int Precision>
    bool is1dsolution(int n,
        const ap::template_1d_array< amp::ampf<Precision> >& y,
        const ap::template_1d_array< amp::ampf<Precision> >& w,
        amp::ampf<Precision> c)
    {
        bool result;
        int i;
        amp::ampf<Precision> v;
        amp::ampf<Precision> s1;
        amp::ampf<Precision> s2;
        amp::ampf<Precision> s3;
        amp::ampf<Precision> delta;


        delta = amp::ampf<Precision>("0.001");
        
        //
        // Test result
        //
        s1 = 0;
        for(i=0; i<=n-1; i++)
        {
            s1 = s1+amp::sqr<Precision>(w(i)*(c-y(i)));
        }
        s2 = 0;
        s3 = 0;
        for(i=0; i<=n-1; i++)
        {
            s2 = s2+amp::sqr<Precision>(w(i)*(c+delta-y(i)));
            s3 = s3+amp::sqr<Precision>(w(i)*(c-delta-y(i)));
        }
        result = s2>=s1 && s3>=s1;
        return result;
    }


    /*************************************************************************
    Silent unit test
    *************************************************************************/
    template<unsigned int Precision>
    bool testratinterpolation_test_silent()
    {
        bool result;


        result = testri<Precision>(true);
        return result;
    }


    /*************************************************************************
    Unit test
    *************************************************************************/
    template<unsigned int Precision>
    bool testratinterpolation_test()
    {
        bool result;


        result = testri<Precision>(false);
        return result;
    }
} // namespace

#endif
