#ifndef BITMAP_H
#define BITMAP_H

#include <windows.h>
#include "SmartRef.h"
class Stream;

/**
 * @ingroup Graphics
 The Bitmap class encapsulates a windows Bitmap handle. It allows loading and
 saving of bitmap files and display to the screen. It automatically handles /
 transparency masks.

 The BitmapRef class is provided for convenience, so you will not have to
 worry about freeing the bitmaps. 
*/

class Bitmap
{
public:
    Bitmap();
    ~Bitmap();

    /// Free all memory associated with the bitmap.
    void destroy();
    void copy(Bitmap* other);

    bool createCompatible( HDC dc, int nwidth, int nheight );
    bool createCompatible( HWND hwnd, int nwidth, int nheight );
    bool loadResource( LPCTSTR lpBitmapName );
    operator HBITMAP();
    operator HDC();
    void bitBlt( HDC dc, int XDest, int YDest, int Width, int Height,
            int XSrc, int YSrc);
    void stretchBlt( HDC dc, int XDest, int YDest, int WidthDest, int HeightDest,
            int XSrc, int YSrc, int WidthSrc, int HeightSrc ); 
    void makeTransparent( COLORREF );
    void noTransparent();
    void scale( HDC dc, int w, int h );
    bool load( HDC dc, const TCHAR* fileName );

    bool load( HDC dc, Stream* stream );
    bool save( Stream* stream );
    
    bool hasMask();
    bool loadMask( Stream* stream );
    bool saveMask( Stream* stream );

    COLORREF getPixel( int x, int y );
    bool empty();

    int width();
    int height();

    HDC getDC();

protected:
    void calcSize();
    HBITMAP _hBitmap;
    HBITMAP _hMask;
    HDC _hDC;
    HBITMAP _hOldBitmap;
    SIZE _size;
};

/**
  \class BitmapRef
  The BitmapRef class is provided for convienience. It will automatically free
  the bitmap when it goes out of scope, and keeps track of references when
  copied. However, it is not safe to use with multiple threads. Example:
  
  \code
  BitmapRef bmp = new Bitmap()
  bmp->load( ScreenDC(), "myfile.bmp" )
  \endcode
 */
typedef SmartRef<Bitmap> BitmapRef;

#endif // BITMAP_H
