#include <assert.h>
#include "ButtonBar.h"
#include <windowsx.h>
#include "Bitmap.h"
#include "ButtonWindow.h"
#include "debug.h"

class ButtonBar::Button
{
public:
    Button( HINSTANCE hInstance )
    {
        button = new ButtonWindow( hInstance );  
        window.hwnd = button->hwnd;
    }

    Button( HWND hwnd ) {
        button = 0;
        window.hwnd = hwnd;
    }

    ~Button()
    {
        delete button;
    }

    RECT rect;
    ButtonWindow* button;
    WindowFunctions window;
    unsigned id;
};

#define MAX_BUTTONS 16

/**
 Construct a ButtonBar window.
 */
ButtonBar::ButtonBar(HINSTANCE hInstance)
{
	_hInstance = hInstance;
	_pszClassName = _T("ButtonBar");
	_pszTitle = _T("insert title here");

    _dwStyle |= WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN | WS_CLIPSIBLINGS; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = LoadCursor(NULL, IDC_ARROW );
	_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
    _buttons = new Button*[MAX_BUTTONS];
    _numButtons = 0;
    _padding = 0;
    _spacing = 0;
    _backColour = RGB( 102, 153, 255 );
    _backImage = NULL;
    _backImageRightAlign = false;
}


ButtonBar::~ButtonBar()
{
    int i;
    for ( i = 0; i < _numButtons; i++ ) {
        delete _buttons[i];
    }
    delete[] _buttons;
    delete _backImage;
}

/// Create a ButtonBar window.
void
ButtonBar::create(HWND hwnd, bool reduceSize)
{
    _cmdTarget = hwnd;
    _reduceSize = reduceSize;
    Window::Create( 0, 0, 10, 10, hwnd, NULL, _hInstance );
}

/// Sets the target window for ButtonBar WM_COMMAND notifications.
void 
ButtonBar::setCommandTarget( HWND hTarget )
{
    _cmdTarget = hTarget;
}

LRESULT 
ButtonBar::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}

void 
ButtonBar::scale( BitmapRef bmp )
{
    if ( !_reduceSize ) {
        return;
    }

    double width = bmp->width();
    double height = bmp->height();
    HDC dc = GetDC(NULL);
    bmp->scale( dc, (int)(width * 0.8), (int)(height * 0.8) );
    ReleaseDC( NULL, dc );        
}

void
ButtonBar::addButton( HWND hwnd )
{
    if ( _numButtons == MAX_BUTTONS ) {
        assert(0);
        return;
    }

    Button* button = new Button(hwnd);
    _buttons[_numButtons++] = button;
    layout();
}

void 
ButtonBar::addButton( unsigned id, const _TCHAR* bmp_out_id, 
    const _TCHAR* bmp_over_id, const _TCHAR* bmp_down_id, 
    const _TCHAR* bmp_disabled_id )
{
    if ( _numButtons == MAX_BUTTONS ) {
        assert(0);
        return;
    }

    Button* button = new Button(_hInstance);
    button->button->create(hwnd, id);
    button->window.hwnd = button->button->hwnd;
    button->id = id;
    BitmapRef bmp;

    if ( bmp_out_id ) {
        bmp = new Bitmap();
        bmp->loadResource( bmp_out_id );
        scale(bmp);
        button->button->setBitmap( ButtonWindow::Out, bmp );
    }

    if ( bmp_over_id ) {
        bmp = new Bitmap();
        bmp->loadResource( bmp_over_id );
        scale(bmp);
        button->button->setBitmap( ButtonWindow::Over, bmp );
    }

    if ( bmp_down_id ) {
        bmp = new Bitmap();
        bmp->loadResource( bmp_down_id );
        scale(bmp);
        button->button->setBitmap( ButtonWindow::Down, bmp );
    }

    if ( bmp_disabled_id ) {
        bmp = new Bitmap();
        bmp->loadResource( bmp_disabled_id );
        scale(bmp);
        button->button->setBitmap( ButtonWindow::Disabled, bmp );
    }

    _buttons[_numButtons++] = button;
    layout();
}

/**
  Adds a button to the ButtonBar. 
 
 @param id The command identifier of the button, to be used in WM_COMMAND notifications
 @param bmp_out_id The resource file identifier of the bitmap to be displayed
      normally in the button.
 @param bmp_over_id The resource file identifier of the bitmap to be displayed
      when the mouse is over the button
 @param bmp_down_id The resource file identifier of the bitmap to be displayed
      when the mouse is over the button and the left button is depressed.
 @param bmp_disable_id The resource file identifier of the bitmap to be displayed
      when the button is disabled.

 Any of bmp_out_id, bmp_over_id, bmp_down_id, bmp_disabled_id may be set to 0
 if not required.
 */
void 
ButtonBar::addButton( unsigned id, unsigned bmp_out_id, unsigned bmp_over_id,
        unsigned bmp_down_id, unsigned bmp_disabled_id )
{
    addButton( id, MAKEINTRESOURCE(bmp_out_id), 
               MAKEINTRESOURCE(bmp_over_id), MAKEINTRESOURCE(bmp_down_id),
               MAKEINTRESOURCE(bmp_disabled_id) );
}

ButtonBar::Button*
ButtonBar::find( unsigned id )
{
    int i = 0;
    for ( i = 0; i < _numButtons; i++ ) {
        if ( _buttons[i]->id == id ) {
            return _buttons[i];
        }
    }

    return 0;
}

/// Enable or disable the given Button. A disabled button will show the disabled image, and it is
/// not clickable.
void 
ButtonBar::enable( unsigned id, bool enabled )
{
    Button* button = find( id );
    if ( button ) {
        button->button->setEnabled( enabled );
    } else {
        assert(0);
    }
}

bool
ButtonBar::isHorizontal()
{
    RECT crect = getClientRect();
    return crect.bottom < crect.right;
}

void
ButtonBar::layout()
{
    int i;
    int pos = _padding;
    RECT crect = getClientRect();
    bool horizontal = crect.bottom < crect.right;
    int width = crect.right - crect.left;
    int height = crect.bottom;
    for ( i = 0; i < _numButtons; i++ ) {
        RECT rect = _buttons[i]->window.getClientRect();
        int x, y;
        if ( horizontal ) {
            y = height / 2 - (rect.bottom - rect.top)/2;
            x = pos;
        } else {
            y = pos;
            x = width / 2 - (rect.right - rect.left)/2;
        }
        _buttons[i]->window.moveWindow( 
                x, y, rect.right - rect.left, rect.bottom - rect.top, TRUE );

        if ( horizontal ) {
            pos = x + rect.right + _spacing;
        } else {
            pos = y + rect.bottom + _spacing;
        }
    }

}

void
ButtonBar::onClose(HWND)
{

}

void 
ButtonBar::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
   // if ( codeNotify == BN_CLICKED ) {
        SendMessage( _cmdTarget, WM_COMMAND, MAKEWPARAM( id, BN_CLICKED ),
                (LPARAM)hwnd );
  //  }

}

BOOL 
ButtonBar::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{

    return TRUE;
}

void
ButtonBar::onDestroy(HWND hwnd)
{

}


void 
ButtonBar::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
ButtonBar::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{

}

void 
ButtonBar::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
ButtonBar::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{

}

void
ButtonBar::setBackColour( DWORD colour ) 
{
    _backColour = colour;
}

void 
ButtonBar::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;
    RECT rect = getClientRect();

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    Brush backBrush( _backColour );
    FillRect( hdc, &rect, backBrush );

    if ( _backImage ) {
        int x = 0;
        int y = 0;
        if ( _backImageRightAlign ) {
            x = rect.right - _backImage->width();
        }
        _backImage->bitBlt( hdc, x, y, _backImage->width(), _backImage->height(), 0, 0 );
    }

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
ButtonBar::onSize(HWND hwnd, UINT state, int cx, int cy)
{
    layout();
}

void 
ButtonBar::setBackImage( const _TCHAR* bmp_id, bool rightAlign )
{
    Bitmap* bmp = new Bitmap();
    if ( !bmp->loadResource( bmp_id ) ) {
        assert(0);
        delete bmp;
        return;
    }

    _backImage = bmp;
    _backImageRightAlign = rightAlign;
}


