#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <assert.h>
#include "ButtonWindow.h"

/**
 Construct a ButtonWindow.

 @param hInstance The module instance obtained in WinMain()
 */
ButtonWindow::ButtonWindow(HINSTANCE hInstance)
{
	_hInstance = hInstance;
	_pszClassName = TEXT("ButtonWindow");
	_pszTitle = TEXT("Button Window");

    _dwStyle |= WS_VISIBLE | WS_CHILD; 
 
    _WndClass.hbrBackground = (HBRUSH)NULL_BRUSH;
    _WndClass.hCursor = LoadCursor( NULL, IDC_ARROW );
	//_WndClass.style |= CS_HREDRAW | CS_VREDRAW;
    _dwExtendedStyle |= 0;
    _state = Out;
    _id = 0;
    _down = false;
}


ButtonWindow::~ButtonWindow()
{

}

/**
 Create a ButtonWindow. A ButtonWindow must be created before it can be used.

 @param hParent Parent window

 @param id The command identifier for the button. When clicked, the button will
 send a WM_COMMAND message to its parent with this command identifier.

 */
void
ButtonWindow::create(HWND hParent, unsigned id)
{
    if ( NULL == Window::Create( CW_USEDEFAULT, CW_USEDEFAULT, 
                CW_USEDEFAULT, CW_USEDEFAULT, hParent, 0, _hInstance ) ) {
        DWORD dwErr = GetLastError();
        assert( false );
    }
    _id = id;

}

/**
  Sets a bitmap associated with a button state. The button will automatically
  size itself so that the largest bitmap will fit. However, it looks best if all
  the bitmaps are the same size.

  @param type The type of the bitmap. Must be one of ButtonWindow::Disabled,
  ButtonWindow::Out, ButtonWindow::Over, or ButtonWindow:Down.

  @param bmp A BitmapRef pointing to a Bitmap object.
 */
void 
ButtonWindow::setBitmap( ButtonWindow::BitmapType_e type, BitmapRef bmp )
{
    if ( type >= NUM_BITMAPS ) {
        assert(0);
        return;
    }

    _bmps[type] = bmp;

    setSize();

    if ( _state == type ) {
        InvalidateRect( hwnd, NULL, TRUE );
    }

}

LRESULT 
ButtonWindow::WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed)
{
    *pbProcessed = TRUE;

    switch(msg)
    {
        HANDLE_MSG(hwnd, WM_CLOSE, onClose);
        HANDLE_MSG(hwnd, WM_COMMAND, onCommand);
        HANDLE_MSG(hwnd, WM_CREATE, onCreate);
        HANDLE_MSG(hwnd, WM_DESTROY, onDestroy);          
        HANDLE_MSG(hwnd, WM_KEYDOWN, onKey);
        HANDLE_MSG(hwnd, WM_LBUTTONDOWN, onLButtonDown);
        HANDLE_MSG(hwnd, WM_LBUTTONUP, onLButtonUp);
        HANDLE_MSG(hwnd, WM_MOUSEMOVE, onMouseMove);
        HANDLE_MSG(hwnd, WM_SIZE, onSize);
        case WM_MOUSELEAVE:
            onMouseLeave(hwnd);
            return 0;
        case WM_ERASEBKGND:
            return 1;
        case WM_PRINTCLIENT:
            onPaint(hwnd, wParam, lParam, true);  
            return 0;
        case WM_PAINT:
            onPaint(hwnd, wParam, lParam, false);
            return 0;
    }
    
    *pbProcessed = FALSE;
    return 0;
}


void
ButtonWindow::onClose(HWND)
{

}

/**
 Disables or enabled the button. When disabled, the disabled bitmap will be
 shown if it has been set, and the button will not be clickable.
 */
void
ButtonWindow::setEnabled( bool enabled )
{
    if ( enabled ) {
        _state = Out;
    } else {
        _state = Disabled;
    }
    InvalidateRect( hwnd, NULL, TRUE );
}


void 
ButtonWindow::onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{

}

BOOL 
ButtonWindow::onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct)
{

    return TRUE;
}

void
ButtonWindow::onDestroy(HWND hwnd)
{

}


void 
ButtonWindow::onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{


}

void 
ButtonWindow::onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, 
                               UINT keyFlags)
{
    if ( _state == Disabled ) {
        return;
    }
    _down = true;
    _state = Down;
    InvalidateRect( hwnd, NULL, TRUE );
}

void 
ButtonWindow::onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( _state != Disabled ) {
        _down = false;
        _state = Over;
        InvalidateRect( hwnd, NULL, TRUE );
    }

	HWND hParent = (HWND)GetWindowLong( hwnd, GWL_HWNDPARENT );
    if ( hParent ) {
        SendMessage( hParent, WM_COMMAND, MAKEWPARAM( _id, BN_CLICKED ), (LPARAM)hwnd );
    }

}

void
ButtonWindow::onMouseLeave( HWND hwnd )
{
    if ( _state == Disabled ) {
        return;
    }
    _state = Out;
    InvalidateRect( hwnd, NULL, TRUE );
}

void
ButtonWindow::onMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
    if ( _state == Disabled ) {
        return;
    }

    _down = (keyFlags & MK_LBUTTON) != 0;

    if ( _state != Over ) {
        _state = _down ? Down : Over;
        TRACKMOUSEEVENT tme;
        memset( &tme, 0, sizeof(tme) );
        tme.cbSize = sizeof(tme);
        tme.dwFlags = TME_LEAVE;
        tme.hwndTrack = hwnd;
        tme.dwHoverTime = 0;
        _TrackMouseEvent( &tme );
        InvalidateRect( hwnd, NULL, TRUE );
    }
}

void 
ButtonWindow::onPaint( HWND hwnd, WPARAM wParam, LPARAM lParam, bool printClient )
{
	PAINTSTRUCT PaintStruct;
    HDC hdc;

    if ( !printClient ) {
	    BeginPaint(hwnd, &PaintStruct);
        hdc = PaintStruct.hdc;
    } else {
        hdc = (HDC)wParam;
    }

    BitmapRef bmp;
    int i;
    for ( i = _state; i >= 0; i-- ) {
        if ( _bmps[i] ) {
            bmp = _bmps[i];
            break;
        }
    }

    if ( bmp ) {
        bmp->bitBlt( hdc, 0, 0, bmp->width(), bmp->height(), 0, 0 );
    }

    if ( !printClient ) {
	    EndPaint(hwnd, &PaintStruct);
    }
}


void 
ButtonWindow::onSize(HWND hwnd, UINT state, int cx, int cy)
{

}

void
ButtonWindow::setSize()
{
     int i;
     int x = 12000, y = 12000;
     for ( i = 0; i < NUM_BITMAPS; i++ ) {
         if ( _bmps[i] ) {
             if ( _bmps[i]->width() < x ) {
                 x = _bmps[i]->width();
             }
             if ( _bmps[i]->height() < y ) {
                 y = _bmps[i]->height();
             }
         }
     }

     RECT rect = getClientRect();
     if ( rect.right - rect.left != x || rect.bottom - rect.top != y ) {
         sizeWindow( x, y );
     }
}
