#ifndef CARDS_H
#define CARDS_H

#include "tstring"
#include "BaseWnd.h"
#include <map>
#include <list>
#include <vector>
#include "Bitmap.h"

class CTag;
class CardStyle;
class CardElement;
class CardParagraph;
class CardMouseEvent;
class CardEventListener;
class CardElementLineSpan;

class CardEventListener {
public:
    virtual ~CardEventListener();
    virtual void onCardEvent( const std::string& event )=0; 
};

class CardDocument : public CardEventListener
{
public:
    CardDocument();
    ~CardDocument();
    bool loadFromString( HWND hwnd, const std::_tstring& text );
    bool loadFromFile( HWND hwnd, const std::_tstring& file );

    int format( DeviceContext dc, int x, int y, int width );
    void draw( DeviceContext dc, int xoffset, int yoffset );
    void clear();
    void processMouseEvent( CardMouseEvent* event );
    void setListener( CardEventListener* listener );
    void showCard( const std::string& card );
    void show( const std::string& name, bool visible );
    virtual void onCardEvent( const std::string& event );
private:
    void destruct();
    void parseCommon( CardElement* element, CTag* tag );
    void parseTag( CardElement* parent, CTag* tag );
    void parseDoc( CardElement* parent, CTag* tag );
    void parseStyle( CardElement* parent, CTag* tag );
    void parseCard( CardElement* parent, CTag* tag );
    void parseP( CardElement* parent, CTag* tag );
    void parseText( CardElement* parent, CTag* tag );
    void parseButton( CardElement* parent, CTag* tag );
    void parseInput( CardElement* parent, CTag* tag );
    typedef std::map<std::string, CardStyle*> StyleMap;
    StyleMap _styles;

    typedef std::map<std::string, CardElement*> NameMap;
    NameMap _named;

    CardElement* _root;
    CardEventListener* _listener;
    typedef std::list<std::string> StringList;
    StringList _eventQueue;
    HWND _hwnd;
};

class CardStyle
{
public:
    CardStyle( CardStyle* parent );
    ~CardStyle();

    bool lookup( const std::string& key, std::string* value, const std::string& def );
    bool lookup( const std::string& key, int* value, int def );
    bool lookup( const std::string& key, bool* value, bool def );
    bool lookupColour( const std::string& key, DWORD* value, DWORD def );

    void set( const std::string& key, const std::string& value );
private:
    typedef std::list<CardStyle*> List;
    typedef std::map<std::string, std::string> Map;

    CardStyle* _parent;
    Map _map;
};


class CardElement
{
public:
    CardElement( CardElement* parent );
    virtual ~CardElement();

    virtual void processMouseEvent( CardMouseEvent* event );

    virtual void draw( DeviceContext dc, int offsetX, int offsetY );
    virtual void flow( DeviceContext dc, CardParagraph* paragraph, int width );

    void moveTo( int x, int y );
    int  layout( int );
    RECT getRelativeRect();
    RECT getAbsoluteRect();
    bool isInline();
    CardElement* getParent();
    bool visible();
    void offset( int* x, int* y);
    void unoffset( int* x, int* y);
    void offset( RECT* rect );
    void sendEvent( const std::string& event );
    void sendClickEvent();
    virtual int format( DeviceContext dc, int x, int y, int width );

    typedef std::list<CardElement*> List;
    
    List children;
    CardStyle* _style;
    CardElement* _root;
    CardEventListener* _listener;
    std::string onclick;
    bool _visible;
    bool _isCard;

    int _x;
    int _y;
    int _width;
    int _height;

    int _requestedWidth;
    int _requestedHeight;

    int _marginLeft;
    int _marginTop;
    int _marginRight;
    int _marginBottom;

protected:
    CardElement* _parent;
    friend class CardElementLineSpan;
    bool _isinline;

};

class FadeTimer;

class FadingCardElement : public CardElement
{
public:
    FadingCardElement(CardElement* parent);
    virtual ~FadingCardElement();

    virtual void setFadeTime( unsigned TimeMs );
    void startFading( HWND hwnd, COLORREF fromColour, COLORREF toColour );
    bool isFading();
    void onMouseOut();
    void setFadedColour( COLORREF );
    COLORREF getFadeColour();

protected:

private:
    void invalidate();
    friend class FadeTimer;
    bool _fading;
    HWND _hwnd;
        
    void onTimer();
    unsigned _fadeTimeMs;
    FadeTimer* _fadeTimer;
    unsigned _timeStarted;
    double _intensity;
    COLORREF _fromColour;
    COLORREF _toColour;
    COLORREF _fadedColour;
};


class LineSpan {
public:
    int _x;
    int _y;
    int _width;
    int _height;
    virtual void draw( DeviceContext dc, int offsetX, int offsetY )=0;
    virtual void processMouseEvent( CardMouseEvent* event );
    LineSpan();
    LineSpan( int x, int y, int width, int height );
    virtual ~LineSpan();
};

class CardParagraph : public CardElement
{
public:
    CardParagraph( CardElement* parent );
    virtual ~CardParagraph();
    virtual int format( DeviceContext dc, int x, int y, int width );
    virtual  void draw( DeviceContext dc, int offsetX, int offsetY );
    virtual void processMouseEvent( CardMouseEvent* event );

    int getLineCount();
    int getLineWidth(int lineno);
    void getLineEnd( int lineno, int* x, int* y );
    void addToLine( int lineno, LineSpan* );
    void addToLine( int lineno, CardElement* info );
    void newLine();

private:
    void deleteLines();
    struct Line {
        int x;
        int y;
        int width;
        int height;
        std::list<LineSpan*> spans;
    };
    typedef std::vector<Line*> LineList;
    LineList _lines;
};

class CardText : public CardElement
{
public:
    CardText( CardElement* parent );
    virtual ~CardText();
    virtual void flow( DeviceContext dc, CardParagraph* paragraph, int width );

    std::string text;
    friend class CardTextLineSpan;
private:
    void makeLine( CardParagraph* paragraph, int lineNo, int start, int length, int width);
    int split( int start, int width, float* total, bool* graceful );
    void createFont();
    void calcWidths( DeviceContext dc );
    float* _widths;
    Font* _font;
};

class CardButton : public FadingCardElement
{
public:
    CardButton( CardElement* parent );
    virtual ~CardButton();
    virtual void draw( DeviceContext dc, int offsetX, int offsetY );
    virtual int format( DeviceContext dc, int x, int y, int width );
    virtual void processMouseEvent( CardMouseEvent* event );

    std::string text;
private:
    void createFont();
    DWORD _backColour;
    DWORD _backColourHover;
    DWORD _backColourDown;
    DWORD _textColour;
    Font* _font;
    int _buttonPadTop;
    int _buttonPadLeft;
    bool _hover;
    bool _down;
};

class CardMouseEvent
{
public:
    enum Type {
        MOVE, UP, DOWN
    } type;

    CardMouseEvent( HWND hwnd, Type type, int x, int y, bool down );
    void offset( int ox, int oy );
    void invalidate( int x, int y, int width, int height );
    bool inside( int x, int y, int width, int height );
    bool down;

    HWND hwnd;
    int x;
    int y;
    int ox;
    int oy;
};

class CardControl : public CardElement
{
public:
    CardControl( CardElement* parent, HWND hControl );
    ~CardControl();

    virtual int format( DeviceContext dc, int x, int y, int width );
    virtual void draw( DeviceContext dc, int offsetX, int offsetY );

    HWND _hControl;
    int _overlappingHeight;
};

class CardWindowTimer;

class CardWindow : public Window, public CardEventListener
{
public:
    CardWindow(HINSTANCE);
    ~CardWindow();

    void create(HWND hParent);
    bool loadFromFile( const std::_tstring& filename );
    void showCard( const std::string& name );
    void onTimer();
    void setListener( CardEventListener* listener );
    void show( const std::string& name, bool visible, bool redraw = false );
    void format();

    virtual void onCardEvent( const std::string& event );

private:
	virtual LRESULT WindowProc(HWND hwnd, UINT msg, WPARAM wParam,
		LPARAM lParam, PBOOL pbProcessed);

    void onClose(HWND);
    void onCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify);
    BOOL onCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct);
    void onDestroy(HWND hwnd);
    void onKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags);
    void onLButtonDown(HWND hwnd, BOOL fDoubleClick, int x, int y, UINT keyFlags);
    void onLButtonUp(HWND hwnd, int x, int y, UINT keyFlags);
    void onMouseMove(HWND hwnd, int x, int y, UINT keyFlags);
    void onPaint(HWND hwnd, WPARAM wParam, LPARAM lParam, bool printClient);
    void onSize(HWND hwnd, UINT state, int cx, int cy);
    void slideAway();
    void slideIn();
    bool _sliding;
    int _slideAt;
    DWORD _slideStart;
    int _slideFrom;
    int _slideTo;
    CardDocument _doc;
    Bitmap _backBuffer;
    CardWindowTimer* _timer;
    std::string _nextCard;
    CardEventListener* _listener;
};

#endif // CARDS_H
