#include "DibImage.h"
#include "Clipboard.h"
#include "debug.h" // must be last

Clipboard::Clipboard()
{

}

Clipboard::~Clipboard()
{
}
/** Determins if the format is available.
 * @param format The format, defined in the MSDN clipboard documentation.
 */

bool
Clipboard::isFormatAvailable( unsigned format )
{
    return IsClipboardFormatAvailable( format ) == TRUE;
}

/**
 * Opens the clipboard and copies the content into a DibImage object, if the
 * format is an image. All images are converted into 24 bit Device Independent
 * Bitmaps.
 *
 * @param hwnd A top-level window handle. Only one window can open the clipboard
 * at a time, so you need to pass in any window handle.
 */
 
DibImage*
Clipboard::getDib(HWND hwnd)
{
    // NOTE: We don't support DIB's < 24 bit, so we get it as a bitmap and then blit it onto our
    // Dib.
    DibImage* dib = NULL;
    HBITMAP hBitmap = NULL;

    if ( !OpenClipboard(hwnd) ) {
        return NULL;
    }

    hBitmap = (HBITMAP)GetClipboardData( CF_BITMAP );

    dib = new DibImage;
    if ( !dib->createFromBitmap(hBitmap) ) {
        delete dib;
        dib = 0;
        goto error;
    }
    
error:
    CloseClipboard();
    return dib;
}

/**
 * Copies a DibImage to the clipboard.
 *
 * @param hwnd A top-level window handle. Only one window can open the clipboard
 * at a time, so you need to pass in any window handle.
 *
 * @param dib A DibImage object.
 */
bool
Clipboard::copy(HWND hwnd, DibImage* dib )
{
    HGLOBAL hMem = NULL;
    unsigned char* ptr = 0;
    unsigned size;
    unsigned headerSize;
    bool success = false;
    BITMAPINFO* pbmi;

    // OpenClipboard
    if ( !OpenClipboard(hwnd) ) {
        return false;
    }

    // call EmptyClipboard
    if ( !EmptyClipboard() ) {
        goto error;
    }

    // calculate size of the data.
    pbmi = dib->getBitmapInfo();
    size = pbmi->bmiHeader.biSize;
    headerSize = pbmi->bmiHeader.biSize;
    if ( pbmi->bmiHeader.biSizeImage == 0 ) {
        size += pbmi->bmiHeader.biWidth * pbmi->bmiHeader.biHeight * pbmi->bmiHeader.biPlanes *
            pbmi->bmiHeader.biBitCount/8;
    } else {
        size += pbmi->bmiHeader.biSizeImage;
    }

    // Allocate the data using GlobalAlloc with GMEM_MOVEABLE flag.
    hMem = GlobalAlloc( GMEM_MOVEABLE, size );

    if ( hMem == NULL ) {
        goto error;
    }

    ptr = (unsigned char*)GlobalLock( hMem );
    if ( ptr == 0 ) {
        goto error;
    }

    // copy data to clipboard
    memcpy( ptr, pbmi, headerSize );
    memcpy( ptr + pbmi->bmiHeader.biSize, dib->getRawBits(), size-headerSize ); 

    GlobalUnlock( hMem );

    // Call SetClipboardData
    if ( !SetClipboardData( CF_DIB, hMem ) ) {
        goto error;
    }

    hMem = NULL; 

    success = true;
error:
    if ( hMem ) {
        GlobalFree( hMem );
    }
    CloseClipboard();
    return success;
}
