#pragma warning(disable:4786)
#include "ControlPage.h"
#include "DrawTimer.h"
#include <assert.h>

class ControlPos
{
    public:
        Control* control;
        bool hasOffset;
        int x;
        int y;
        RECT end;
        unsigned ticks;
        bool on;
};

class ControlTimer : public TimerEvent
{
public:
    ControlTimer( ControlPage* parent) {
        _parent = parent;
        id = 0;
        _set = false;
    }

    unsigned id;
    bool _set;
    ControlPage* _parent;

    void set(unsigned ms) {
        if ( _set ) {
            DrawTimer::CancelEvent( this );
        }

        id = DrawTimer::RegisterEvent( this );
        _set = true;
    }

    void stop() {
        assert( _set );
        if ( _set ) {
            DrawTimer::CancelEvent( this );
        }
        _set = false;
    }

    ~ControlTimer() {
        DrawTimer::CancelEvent( this ); 
        _parent = 0;
    }

    virtual void fire( unsigned id ) {
        _parent->onTimer();
    }
};


/**
 * Construct a ControlPage. 
 *
 * @param hwnd The parent window.
 */
ControlPage::ControlPage( HWND hwnd ) :
    Control( hwnd )
{
    _controlTimer = new ControlTimer(this);
    _animateTimeMs = Control::DefaultFadeMs;
    _animating = false;

}

ControlPage::~ControlPage()
{
    delete _controlTimer;

    ControlPos* pos = _controlPos.first();

    while( pos ) {
        delete pos;
        pos = _controlPos.next();
    }

    pos = _controlsAll.first();
    while( pos ) {
        delete pos;
        pos = _controlsAll.next();
    }
}

void
ControlPage::animateStop()
{
    if ( _animating ) {
        _controlTimer->stop();
        ControlPos* cs = _controlPos.first();
        while ( cs ) {
            drawAnimated( cs, 1.0 );
            delete cs;
            cs = _controlPos.next();
        }   
        _controlPos.clear();
        _animating = false;
    }
}


/**
 * Begin an animation to slide the controls from their off screen position to
 * the on-screen position.
 *
 * @param on If true, the controls will be animated onto the screen. If false,
 * the controls will be animated off the screen.
 */
void
ControlPage::animate(bool on)
{
    animateStop();
    animateAll( on );
}

void
ControlPage::animate( Control* control, RECT* end, int x, int y, bool on)
{
    ControlPos* cs = new ControlPos;
    cs->control = control;
    cs->end = *end;
    cs->x = x;
    cs->y = y;
    cs->ticks = GetTickCount();
    cs->on = on;

    _controlPos.addTail( cs );

    drawAnimated( cs, 0.0 );
    if ( !_animating ) {
        _animating = true;
        _controlTimer->set( 10 );
    } 
}

void
ControlPage::animateDone(bool show)
{

}

ControlPos*
ControlPage::find( Control* control )
{
    ControlPos* cs = _controlsAll.first();
    while( cs ) {
        if ( cs->control == control ) {
            return cs;
        }
        cs = _controlsAll.next();
    }
    return 0;
}

void
ControlPage::addChild( Control* control, int x, int y, int cx, int cy )
{
    addChild( control, x, y, cx, cy, 0, 0 );
    ControlPos* pos = find( control );
    pos->hasOffset = false;
}

/** 
 * Adds a child control to the page, and sets the on and off-screen positions.
 *
 * @param control The control to add.
 *
 * @param x The x coordinate of the control, relative to the positon of the
 * ControlPage.
 *
 * @param y
 *
 * @param cx The width of the control
 * 
 * @param cy The height of the control
 *
 * @param offsetX The offset in the X position if the control, when it is off
 * screen. This should ideally be set such that it places the control outside
 * of the parent window, where it is not visible.
 *
 * @param offsetY The offset in the Y position of hte control when it is
 * offscreen.
 *
 * @param moveNow If set to true, the control will be moved immediately (not
 * animated) the given onscreen position. If set to false, the control will be
 * moved to the off-screen position.
 */
void 
ControlPage::addChild( Control* control, int x, int y, int cx, int cy, int
        offsetX, int offsetY, bool moveNow )
{
    Control::addChild( control );
    ControlPos* cs = find( control );

    if ( cs == 0 ) {
        cs = new ControlPos;
        _controlsAll.addTail( cs );
    }

    cs->control = control;
    cs->end.left = x;
    cs->end.top = y;
    cs->end.right = x + cx;
    cs->end.bottom = y + cy;
    cs->x = offsetX;
    cs->y = offsetY;
    cs->hasOffset = true;

    if ( moveNow ) {
        control->move( x, y, cx, cy );
    } else {
        control->move( x+offsetX, y+offsetY, cx, cy );
    }
}

void
ControlPage::animateAll( bool on )
{
    ControlPos* cs = _controlsAll.first();
    while( cs ) {
        animate( cs->control, &cs->end, cs->x, cs->y, on );
        cs = _controlsAll.next();
    }
}

void
ControlPage::onTimer()
{
    ControlPos* cs = _controlPos.first();
    bool found = false;
    assert( _animating );
    
    while ( cs ) {
        found = true;
        unsigned diff = GetTickCount() - cs->ticks;

        if ( diff > _animateTimeMs ) {
            drawAnimated( cs, 1.0 );
            delete cs;
            _controlPos.remove();
            goto next;
        }

        drawAnimated( cs, (double)diff / _animateTimeMs );

next:
        cs = _controlPos.next();
    } 

    if ( !found ) {
        _animating = false;
        _controlTimer->stop();
        animateDone( _visible );
    }
    
}

/**
 * Returns true if the control has not set finished performing an animation.
 */
bool
ControlPage::animating()
{
    return _animating;
}

void
ControlPage::drawAnimated(ControlPos* cs, double percent)
{
    Control* control = cs->control;

    if ( cs->on ) {
        percent = 1.0 - percent;
    }

    int x = (int)((double)cs->x * percent) + cs->end.left; 
    int y = (int)((double)cs->y * percent) + cs->end.top; 

    control->move( x, y, cs->end.right - cs->end.left,
            cs->end.bottom - cs->end.top );
    _visible = cs->on;
}

